import { Component, inject, Input, OnInit } from '@angular/core';
import { FormArray, FormBuilder, ReactiveFormsModule, Validators } from '@angular/forms';
import { MatFormFieldModule } from '@angular/material/form-field';
import { MatInputModule } from '@angular/material/input';
import { MatSelectModule } from '@angular/material/select';
import { MatCheckboxModule } from '@angular/material/checkbox';
import { MatButtonModule } from '@angular/material/button';
import { MatIconModule } from '@angular/material/icon';
import { MatDividerModule } from '@angular/material/divider';
import { MatTooltipModule } from '@angular/material/tooltip';
import { CdkDragDrop, DragDropModule, moveItemInArray } from '@angular/cdk/drag-drop';
import { QUESTION_TYPE } from '../models/question-type';
import { TranslateModule } from '@ngx-translate/core';
import { compareOptionValidator } from '../validator/compare-option-validator';
import { MatSlideToggleModule } from '@angular/material/slide-toggle';
import { AUDIT_TEMPLATE_SECTION } from '../models/audit-template-section';
import { ToastrService } from '@core/services/toastr-service';

@Component({
  selector: 'app-question-form',
  standalone: true,
  imports: [
    ReactiveFormsModule,
    MatFormFieldModule,
    MatInputModule,
    MatSelectModule,
    MatCheckboxModule,
    MatButtonModule,
    MatIconModule,
    MatDividerModule,
    MatTooltipModule,
    DragDropModule,
    TranslateModule,
    MatSlideToggleModule,
  ],
  templateUrl: './question-form.component.html',
  styleUrls: ['./question-form.component.scss']
})
export class QuestionFormComponent implements OnInit {
  @Input() questionForm: any;
  @Input() questionTypes: { value: QUESTION_TYPE, label: string }[] = [];
  @Input() sections: { value: AUDIT_TEMPLATE_SECTION, label: string }[] = [];
  ratingType = QUESTION_TYPE.RATING;
  ratingNumbers: string[] = Array.from({ length: 30 }, (_, i) => (i + 1).toString());
  toastrService = inject(ToastrService);

  constructor(private fb: FormBuilder) { }

  ngOnInit(): void {
    // Listen for question type changes
    this.questionForm.get('inputType')?.valueChanges.subscribe((type: any) => {
      this.handleQuestionTypeChange(type);
    });
  }

  handleQuestionTypeChange(type: QUESTION_TYPE): void {
    // If question type has changed to or from a type that needs options
    const needsOptions = type === QUESTION_TYPE.SINGLE_SELECT || type === QUESTION_TYPE.MULTI_SELECT || type === QUESTION_TYPE.RATING;
    // Get the options array
    const optionsArray = this.options;
    this.options.clear();
    // If we need options but have none, add a default option
    if (needsOptions && optionsArray.length === 0) {
      if (type === QUESTION_TYPE.RATING) {
        this.addOptionForRating();
      } else if (type === QUESTION_TYPE.SINGLE_SELECT || type === QUESTION_TYPE.MULTI_SELECT) {
        this.addOptionMultiSelect();
      }
    }
  }

  get options(): FormArray {
    return this.questionForm.get('questionOptions') as FormArray;
  }

  get questionTypeValue(): QUESTION_TYPE {
    return this.questionForm.get('inputType')?.value as QUESTION_TYPE;
  }

  get needsOptions(): boolean {
    const type = this.questionTypeValue;
    return type === QUESTION_TYPE.SINGLE_SELECT || type === QUESTION_TYPE.MULTI_SELECT || type === QUESTION_TYPE.RATING;
  }

  addOptionMultiSelect(): void {
    const newOption = this.fb.group({
      optionText: ['', [Validators.required, Validators.maxLength(50)]],
      optionValue: ['', [Validators.required]],
      orderNo: [this.options.length + 1]
    });

    this.options.push(newOption);
  }
  addOptionForRating(): void {
    const newOption = this.fb.group({
      optionText: [1, [Validators.required, Validators.min(1)]],
      optionValue: [10, [Validators.required]],
      orderNo: [this.options.length + 1]
    }, { validators: compareOptionValidator() });
    this.options.push(newOption);
  }

  removeOption(index: number): void {
    this.options.removeAt(index);
    if (this.options.length <= 1) {
      this.toastrService.error('At least one option is required.');
      return;
    }
    // Update the order for remaining options
    this.updateOptionOrder();
  }

  updateOptionOrder(): void {
    this.options.controls.forEach((control, index: number) => {
      control.get('orderNo')?.setValue(index + 1);
    });
  }

  onOptionDrop(event: CdkDragDrop<string[]>): void {
    moveItemInArray(this.options.controls, event.previousIndex, event.currentIndex);
    this.updateOptionOrder();
  }

  // Helper method to auto-generate a value from the option text
  generateOptionValue(index: number): void {
    const option = this.options.at(index);
    const optionText = option.get('optionText')?.value;

    if (optionText) {
      // Generate a slug-like value
      const value = optionText
        .toLowerCase()
        .replace(/[^a-z0-9]+/g, '_')
        .replace(/^_+|_+$/g, '');

      option.get('value')?.setValue(value);
    }
  }
}
