import { AfterViewInit, Component, inject, OnInit, ViewChild } from '@angular/core';
import { MatPaginator } from '@angular/material/paginator';
import { MatSort, MatSortModule } from '@angular/material/sort';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { Audit } from '@core/domain-classes/audit';
import { PageSizeOption } from '@core/utils/global-config';
import { Subject, debounceTime, distinctUntilChanged, merge, tap } from 'rxjs';
import { DepartmentStore } from '../../audit-template/department/store/department.store';
import { AUDIT_STATUS } from '../../audit-template/models/audit-status';
import { AUDIT_TYPE } from '../../audit-template/models/audit-type';
import { BaseComponent } from '../../base.component';
import { UserStore } from '../../user/store/user.store';
import { FormsModule } from '@angular/forms';
import { MatSelectModule } from '@angular/material/select';
import { MatTableModule } from '@angular/material/table';
import { RouterModule } from '@angular/router';
import { AuditReviewStore } from './audit-review-store';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { TranslateModule } from '@ngx-translate/core';
import { AuditTypePipe } from '../audit-type.pipe';
import { MatButtonModule } from '@angular/material/button';
import { MatCardModule } from '@angular/material/card';
import { MatIconModule } from '@angular/material/icon';
import { NgClass } from '@angular/common';

@Component({
  selector: 'app-audit-for-review-list',
  imports: [
    RouterModule,
    MatTableModule,
    MatPaginator,
    MatSelectModule,
    FormsModule,
    MatSortModule,
    PageHelpTextComponent,
    TranslateModule,
    AuditTypePipe,
    MatButtonModule,
    MatCardModule,
    MatIconModule,
    NgClass
  ],
  templateUrl: './audit-for-review-list.component.html',
  styleUrl: './audit-for-review-list.component.scss'
})
export class AuditForReviewListComponent extends BaseComponent
  implements OnInit, AfterViewInit {
  auditList: Audit[] = [];
  displayedColumns: string[] = [
    'action',
    'title',
    'auditorName',
    'score',
    'auditStatus',
    'auditType',
    'department',
    'nonConformance',
    'capaRequest',
    'auditReviewersComment',
    'auditTemplateName',
    'description',
  ];
  displayedColumnSecondary: string[] = [
    'search-action',
    'search-title',
    'search-auditorName',
    'search-score',
    'search-auditStatus',
    'search-auditType',
    'search-department',
    'search-nonConformance',
    'search-capaRequest',
    'search-auditReviewersComment',
    'search-auditTemplateName',
    'search-description',
  ];
  footerToDisplayed: string[] = ['footer'];
  auditReviewStore = inject(AuditReviewStore);
  pageOption = PageSizeOption;
  userStore = inject(UserStore);
  commandDialogService = inject(CommonDialogService);
  departmentStore = inject(DepartmentStore);
  auditFilterParameter = { ...this.auditReviewStore.filterParameters() };
  filterParameter$: Subject<string> = new Subject<string>();
  _auditorByIdFilter = this.auditFilterParameter.auditorId;
  _departmentIdFilter = this.auditFilterParameter.departmentId;
  _auditStatusFilter = this.auditFilterParameter.auditStatus;
  _auditTypeFilterFilter = this.auditFilterParameter.auditType;
  _auditTemplateNameFilter = this.auditFilterParameter.auditTemplateName;
  _titleFilter = this.auditFilterParameter.title;

  @ViewChild(MatPaginator) paginator: MatPaginator;
  @ViewChild(MatSort) sort: MatSort;
  AUDIT_STATUS = AUDIT_STATUS;
  auditStatusList = Object.keys(AUDIT_STATUS)
    .filter((key) => !isNaN(Number(AUDIT_STATUS[key as any])))
    .map((key) => ({
      label: key,
      value: AUDIT_STATUS[key as keyof typeof AUDIT_STATUS],
    }));

  auditTypes = Object.keys(AUDIT_TYPE)
    .filter((key) => !isNaN(Number(AUDIT_TYPE[key as any]))) // filter numeric values
    .map((key) => ({
      label: key,
      value: AUDIT_TYPE[key as keyof typeof AUDIT_TYPE],
    }));

  public get auditTemplateNameFilter(): string {
    return this._auditTemplateNameFilter;
  }

  public set auditTemplateNameFilter(v: string) {
    if (this._auditTemplateNameFilter !== v) {
      this._auditTemplateNameFilter = v;
      const auditTemplateNameFilter = `auditTemplateName#${v}`;
      this.filterParameter$.next(auditTemplateNameFilter);
    }
  }

  public get titleFilter(): string {
    return this._titleFilter;
  }

  public set titleFilter(v: string) {
    if (this._titleFilter !== v) {
      this._titleFilter = v;
      const titleFilter = `title#${v}`;
      this.filterParameter$.next(titleFilter);
    }
  }

  public get auditorIdFilter(): string | undefined {
    return this._auditorByIdFilter;
  }

  public set auditorIdFilter(v: string) {
    if (this._auditorByIdFilter !== v) {
      this._auditorByIdFilter = v;
      const auditorByIdFilter = `auditorName#${v}`;
      this.filterParameter$.next(auditorByIdFilter);
    }
  }

  public get departmentIdFilter(): string | undefined {
    return this._departmentIdFilter;
  }

  public set departmentIdFilter(v: string) {
    if (this._departmentIdFilter !== v) {
      this._departmentIdFilter = v;
      const departmentIdFilter = `departmentId#${v}`;
      this.filterParameter$.next(departmentIdFilter);
    }
  }

  public get auditStatusFilter(): string {
    return this._auditStatusFilter;
  }

  public set auditStatusFilter(v: string) {
    if (this._auditStatusFilter !== v) {
      this._auditStatusFilter = v;
      const auditStatusFilter = `auditStatus#${v}`;
      this.filterParameter$.next(auditStatusFilter);
    }
  }

  public get auditTypeFilter(): string {
    return this._auditTypeFilterFilter;
  }

  public set auditTypeFilter(v: string) {
    if (this._auditTypeFilterFilter !== v) {
      this._auditTypeFilterFilter = v;
      const auditTypeFilter = `auditType#${v}`;
      this.filterParameter$.next(auditTypeFilter);
    }
  }

  ngOnInit(): void {
    this.getDepartments();
    this.sub$.sink = this.filterParameter$
      .pipe(debounceTime(1000), distinctUntilChanged())
      .subscribe((c: string) => {
        this.auditFilterParameter.skip = 0;
        this.paginator.pageIndex = 0;
        const filterArray: Array<string> = c.split('#');
        if (filterArray[0] === 'auditTemplateName') {
          this.auditFilterParameter.auditTemplateName = filterArray[1];
        } else if (filterArray[0] === 'title') {
          this.auditFilterParameter.title = filterArray[1];
        } else if (filterArray[0] === 'auditorName') {
          this.auditFilterParameter.auditorId = filterArray[1];
        } else if (filterArray[0] === 'auditStatus') {
          this.auditFilterParameter.auditStatus = filterArray[1];
        } else if (filterArray[0] === 'auditType') {
          this.auditFilterParameter.auditType = filterArray[1];
        } else if (filterArray[0] === 'departmentId') {
          this.auditFilterParameter.departmentId = filterArray[1];
        }
        this.auditReviewStore.loadByQuery(this.auditFilterParameter);
      });
  }

  getDepartments(): void {
    this.departmentStore.loadDepartments();
  }

  refresh() {
    this.auditReviewStore.loadByQuery(this.auditFilterParameter);
  }

  ngAfterViewInit(): void {
    this.sub$.sink = this.sort.sortChange.subscribe(() => {
      this.paginator.pageIndex = 0;
    });
    this.sub$.sink = merge(this.sort.sortChange, this.paginator.page)
      .pipe(
        tap(() => {
          this.auditFilterParameter.skip =
            this.paginator.pageIndex * this.paginator.pageSize;
          this.auditFilterParameter.pageSize = this.paginator.pageSize;
          this.auditFilterParameter.orderBy =
            this.sort.active + ' ' + this.sort.direction;
          this.auditReviewStore.loadByQuery(this.auditFilterParameter);
        })
      )
      .subscribe();
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.auditReviewStore.auditList().indexOf(row);
  }
}
