import {
  patchState,
  signalStore,
  withHooks,
  withMethods,
  withState,
} from '@ngrx/signals';
import { rxMethod } from '@ngrx/signals/rxjs-interop';
import { inject } from '@angular/core';

import { ToastrService } from '@core/services/toastr-service';
import { TranslationService } from '@core/services/translation.service';
import { pipe, switchMap, tap } from 'rxjs';
import { tapResponse } from '@ngrx/operators';
import { toObservable } from '@angular/core/rxjs-interop';
import { HttpResponse } from '@angular/common/http';
import { Audit } from '@core/domain-classes/audit';
import { Router } from '@angular/router';
import { AuditResource } from '../audit-resource';
import { AuditService } from '../audit.service';

type AuditReviewState = {
  auditList: Audit[];
  audit: Audit;
  loadList: boolean;
  isAddUpdate: boolean;
  filterParameters: AuditResource;
};

export const initialAuditReviewState: AuditReviewState = {
  auditList: [],
  audit: {} as Audit,
  loadList: true,
  isAddUpdate: false,
  filterParameters: {
    orderBy: 'auditTemplateName asc',
    pageSize: 10,
    skip: 0,
    totalCount: 0,
    auditTemplateName: '',
    title: '',
    auditStatus: '',
    auditorName: '',
    auditType: '',
    departmentId: '',
    auditorId: '',
  },
};

export const AuditReviewStore = signalStore(
  { providedIn: 'root' },
  withState(initialAuditReviewState),
  withMethods(
    (
      store,
      auditService = inject(AuditService),
      router = inject(Router),
      toastrService = inject(ToastrService),
      translationService = inject(TranslationService)
    ) => ({
      loadByQuery: rxMethod<AuditResource>(
        pipe(
          tap(() => patchState(store, { auditList: [] })),
          switchMap((filter) => {
            return auditService.getAuditForReviews(filter).pipe(
              tapResponse({
                next: (httpResponse: HttpResponse<Audit[]>) => {
                  if (httpResponse && httpResponse.body) {
                    patchState(store, {
                      auditList: [...httpResponse.body],
                      loadList: false,
                    });
                  }
                  if (httpResponse.headers.get('x-pagination')) {
                    const pagination: AuditResource = JSON.parse(
                      httpResponse.headers.get('x-pagination') ?? ''
                    );
                    patchState(store, {
                      filterParameters: {
                        ...filter,
                        totalCount: pagination.totalCount,
                      },
                    });
                  }
                },
                error: (error: any) => {
                  toastrService.error(error.error);
                },
              })
            );
          })
        )
      ),
      resetflag() {
        patchState(store, { isAddUpdate: false });
      },
    })
  ),
  withHooks({
    onInit(store) {
      toObservable(store.loadList).subscribe((flag) => {
        if (flag) {
          store.loadByQuery(store.filterParameters());
        }
      });
    },
  })
);
