import {
  AfterViewInit,
  ChangeDetectorRef,
  Component,
  inject,
  OnInit,
  viewChild,
  ViewChild,
} from '@angular/core';
import { FormsModule } from '@angular/forms';
import { MatIconModule } from '@angular/material/icon';
import { MatMenuModule } from '@angular/material/menu';
import { MatPaginator } from '@angular/material/paginator';
import { MatSelectModule } from '@angular/material/select';
import { MatSort, MatSortModule } from '@angular/material/sort';
import { MatTableModule } from '@angular/material/table';
import { Router, RouterModule } from '@angular/router';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { Audit, StartAudit } from '@core/domain-classes/audit';
import { PageSizeOption } from '@core/utils/global-config';
import { debounceTime, distinctUntilChanged, merge, Subject, tap } from 'rxjs';
import { BaseComponent } from '../../base.component';
import { AuditStore } from '../audit-store';
import { DepartmentStore } from '../../audit-template/department/store/department.store';
import { AUDIT_TYPE } from '../../audit-template/models/audit-type';
import { AuditTypePipe } from '../audit-type.pipe';
import { AUDIT_STATUS } from '../../audit-template/models/audit-status';
import { UserStore } from '../../user/store/user.store';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { TranslateModule } from '@ngx-translate/core';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { AuditStatusPipe } from '../audit-status.pipe';
import { animate, state, style, transition, trigger } from '@angular/animations';
import { AuditReviewersComponent } from '../audit-reviewers/audit-reviewers.component';
import { MatButtonModule } from '@angular/material/button';
import { MatCardModule } from '@angular/material/card';
import { NgClass } from '@angular/common';
import { LimitToPipe } from "../../shared/pipes/limit-to.pipe";
import { MatTooltipModule } from '@angular/material/tooltip';
import { AuditReportComponent } from "../audit-report/audit-report.component";

@Component({
  selector: 'app-audit-list',
  imports: [
    RouterModule,
    MatTableModule,
    MatMenuModule,
    MatIconModule,
    MatPaginator,
    MatSelectModule,
    AuditTypePipe,
    FormsModule,
    MatSortModule,
    PageHelpTextComponent,
    TranslateModule,
    HasClaimDirective,
    AuditStatusPipe,
    AuditReviewersComponent,
    MatButtonModule,
    MatCardModule,
    NgClass,
    LimitToPipe,
    MatTooltipModule,
    AuditReportComponent
  ],
  templateUrl: './audit-list.component.html',
  styleUrl: './audit-list.component.scss',
  animations: [
    trigger('detailExpand', [
      state('collapsed', style({ height: '0px', minHeight: '0' })),
      state('expanded', style({ height: '*' })),
      transition('expanded <=> collapsed',
        animate('225ms cubic-bezier(0.4, 0.0, 0.2, 1)')
      ),
    ]),
  ]
})
export class AuditListComponent
  extends BaseComponent
  implements OnInit, AfterViewInit {
  auditList: Audit[] = [];
  AUDIT_STATUS = AUDIT_STATUS;
  displayedColumns: string[] = [
    'action',
    'auditNumber',
    'title',
    'auditorName',
    'score',
    'auditStatus',
    'auditType',
    'department',
    'nonConformance',
    'capaRequest',
    'auditReviewComments',
    'auditTemplateName',
    'description',
  ];
  displayedColumnSecondary: string[] = [
    'search-action',
    'search-auditNumber',
    'search-title',
    'search-auditorName',
    'search-score',
    'search-auditStatus',
    'search-auditType',
    'search-department',
    'search-nonConformance',
    'search-capaRequest',
    'search-auditReviewComments',
    'search-auditTemplateName',
    'search-description',
  ];
  footerToDisplayed: string[] = ['footer'];
  auditStore = inject(AuditStore);
  pageOption = PageSizeOption;
  userStore = inject(UserStore);
  commandDialogService = inject(CommonDialogService);
  departmentStore = inject(DepartmentStore);
  auditFilterParameter = { ...this.auditStore.filterParameters() };
  filterParameter$: Subject<string> = new Subject<string>();
  _auditorByIdFilter = this.auditFilterParameter.auditorId;
  _departmentIdFilter = this.auditFilterParameter.departmentId;
  _auditStatusFilter = this.auditFilterParameter.auditStatus;
  _auditTypeFilterFilter = this.auditFilterParameter.auditType;
  _auditTemplateNameFilter = this.auditFilterParameter.auditTemplateName;
  _titleFilter = this.auditFilterParameter.title;
  router = inject(Router);
  expandedElement: Audit | null;
  isDownloading = false;
  cd = inject(ChangeDetectorRef);
  auditId: string = '';

  @ViewChild(MatPaginator) paginator: MatPaginator;
  @ViewChild(MatSort) sort: MatSort;

  auditStatusList = Object.keys(AUDIT_STATUS)
    .filter((key) => !isNaN(Number(AUDIT_STATUS[key as any])))
    .map((key) => ({
      label: key,
      value: AUDIT_STATUS[key as keyof typeof AUDIT_STATUS],
    }));

  auditTypes = Object.keys(AUDIT_TYPE)
    .filter((key) => !isNaN(Number(AUDIT_TYPE[key as any]))) // filter numeric values
    .map((key) => ({
      label: key,
      value: AUDIT_TYPE[key as keyof typeof AUDIT_TYPE],
    }));

  public get auditTemplateNameFilter(): string {
    return this._auditTemplateNameFilter;
  }

  public set auditTemplateNameFilter(v: string) {
    if (this._auditTemplateNameFilter !== v) {
      this._auditTemplateNameFilter = v;
      const auditTemplateNameFilter = `auditTemplateName#${v}`;
      this.filterParameter$.next(auditTemplateNameFilter);
    }
  }

  public get titleFilter(): string {
    return this._titleFilter;
  }

  public set titleFilter(v: string) {
    if (this._titleFilter !== v) {
      this._titleFilter = v;
      const titleFilter = `title#${v}`;
      this.filterParameter$.next(titleFilter);
    }
  }

  public get auditorIdFilter(): string {
    return this._auditorByIdFilter ?? '';
  }

  public set auditorIdFilter(v: string) {
    if (this._auditorByIdFilter !== v) {
      this._auditorByIdFilter = v;
      const auditorByIdFilter = `auditorName#${v}`;
      this.filterParameter$.next(auditorByIdFilter);
    }
  }

  public get departmentIdFilter(): string {
    return this._departmentIdFilter ?? '';
  }

  public set departmentIdFilter(v: string) {
    if (this._departmentIdFilter !== v) {
      this._departmentIdFilter = v;
      const departmentIdFilter = `departmentId#${v}`;
      this.filterParameter$.next(departmentIdFilter);
    }
  }

  public get auditStatusFilter(): string {
    return this._auditStatusFilter;
  }

  public set auditStatusFilter(v: string) {
    if (this._auditStatusFilter !== v) {
      this._auditStatusFilter = v;
      const auditStatusFilter = `auditStatus#${v}`;
      this.filterParameter$.next(auditStatusFilter);
    }
  }

  public get auditTypeFilter(): string {
    return this._auditTypeFilterFilter;
  }

  public set auditTypeFilter(v: string) {
    if (this._auditTypeFilterFilter !== v) {
      this._auditTypeFilterFilter = v;
      const auditTypeFilter = `auditType#${v}`;
      this.filterParameter$.next(auditTypeFilter);
    }
  }

  ngOnInit(): void {
    this.getDepartments();
    this.sub$.sink = this.filterParameter$
      .pipe(debounceTime(1000), distinctUntilChanged())
      .subscribe((c: string) => {
        this.auditFilterParameter.skip = 0;
        this.paginator.pageIndex = 0;
        const filterArray: Array<string> = c.split('#');
        if (filterArray[0] === 'auditTemplateName') {
          this.auditFilterParameter.auditTemplateName = filterArray[1];
        } else if (filterArray[0] === 'title') {
          this.auditFilterParameter.title = filterArray[1];
        } else if (filterArray[0] === 'auditorName') {
          this.auditFilterParameter.auditorId = filterArray[1];
        } else if (filterArray[0] === 'auditStatus') {
          this.auditFilterParameter.auditStatus = filterArray[1];
        } else if (filterArray[0] === 'auditType') {
          this.auditFilterParameter.auditType = filterArray[1];
        } else if (filterArray[0] === 'departmentId') {
          this.auditFilterParameter.departmentId = filterArray[1];
        }
        this.auditStore.loadByQuery(this.auditFilterParameter);
      });
  }

  addAudit() {
    this.router.navigate(['audit/audit-template']);
  }

  getDepartments(): void {
    this.departmentStore.loadDepartments();
  }

  refresh() {
    this.auditStore.loadByQuery(this.auditFilterParameter);
  }

  ngAfterViewInit(): void {
    this.sub$.sink = this.sort.sortChange.subscribe(() => {
      this.paginator.pageIndex = 0;
    });
    this.sub$.sink = merge(this.sort.sortChange, this.paginator.page)
      .pipe(
        tap(() => {
          this.auditFilterParameter.skip =
            this.paginator.pageIndex * this.paginator.pageSize;
          this.auditFilterParameter.pageSize = this.paginator.pageSize;
          this.auditFilterParameter.orderBy =
            this.sort.active + ' ' + this.sort.direction;
          this.auditStore.loadByQuery(this.auditFilterParameter);
        })
      )
      .subscribe();
  }

  closeAudit(audit: Audit) {
    this.commandDialogService.deleteConfirmtionDialog(this.translationService.getValue("ARE_YOU_SURE_YOU_WANT_TO_CLOSE_AUDIT")).subscribe({
      next: (result: boolean) => {
        if (result) {
          const startAudit: StartAudit = {
            id: audit.id || '',
            auditTemplateId: audit?.auditTemplateId || '',
            status: AUDIT_STATUS.CLOSED,
          };
          this.auditStore.updateAudit(startAudit);
        }
      }
    });
  }

  toggleRow(audit: Audit) {
    this.expandedElement = this.expandedElement === audit ? null : audit;
    this.cd.detectChanges();
  }

  downloadAuditReport(auditId: string) {
    this.isDownloading = true;
    this.auditId = auditId;
  }

  onDownloadComplete() {
    this.isDownloading = false;
    this.auditId = '';
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.auditStore.auditList().indexOf(row);
  }
}
