import {
  AfterViewInit,
  Component,
  inject,
  OnInit,
  ViewChild,
} from '@angular/core';
import { MatPaginator } from '@angular/material/paginator';
import { MatSort, MatSortModule } from '@angular/material/sort';
import { Audit } from '@core/domain-classes/audit';
import { PageSizeOption } from '@core/utils/global-config';
import { Subject, debounceTime, distinctUntilChanged, merge, tap } from 'rxjs';
import { DepartmentStore } from '../../audit-template/department/store/department.store';
import { AUDIT_STATUS } from '../../audit-template/models/audit-status';
import { UserStore } from '../../user/store/user.store';
import { AuditLogService } from './audit-log.service';
import { AuditResource } from '../audit-resource';
import { BaseComponent } from '../../base.component';
import { HttpResponse } from '@angular/common/http';
import { MatTableModule } from '@angular/material/table';
import { MatSelectModule } from '@angular/material/select';
import { FormsModule } from '@angular/forms';
import { AUDIT_LOG_STATUS } from '../models/audit-log-status.enum';
import { RouterModule } from '@angular/router';
import { AuditResponseLog } from '@core/domain-classes/audit-response-log';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { TranslateModule } from '@ngx-translate/core';
import { MatCardModule } from '@angular/material/card';
import { LimitToPipe } from "../../shared/pipes/limit-to.pipe";
import { MatTooltip } from '@angular/material/tooltip';
import { DatePipe, NgClass } from '@angular/common';

@Component({
  selector: 'app-audit-log-list',
  imports: [
    RouterModule,
    MatTableModule,
    MatPaginator,
    MatSelectModule,
    FormsModule,
    MatSortModule,
    PageHelpTextComponent,
    TranslateModule,
    MatCardModule,
    LimitToPipe,
    MatTooltip,
    NgClass,
    DatePipe
  ],
  templateUrl: './audit-log-list.component.html',
  styleUrl: './audit-log-list.component.scss',
})
export class AuditLogListComponent
  extends BaseComponent
  implements OnInit, AfterViewInit {
  auditList: AuditResponseLog[] = [];
  displayedColumns: string[] = [
    'auditDate',
    'title',
    'auditorName',
    'score',
    'auditStatus',
    'department',
    'question',
    'response',
    'logStatus',
  ];
  displayedColumnSecondary: string[] = [
    'search-auditDate',
    'search-title',
    'search-auditorName',
    'search-score',
    'search-auditStatus',
    'search-department',
    'search-question',
    'search-response',
    'search-logStatus',
  ];
  footerToDisplayed: string[] = ['footer'];
  auditLogService = inject(AuditLogService);
  pageOption = PageSizeOption;
  userStore = inject(UserStore);
  departmentStore = inject(DepartmentStore);
  auditLogList: Audit[] = [];
  AUDIT_STATUS = AUDIT_STATUS;

  auditFilterParameter: AuditResource = {
    skip: 0,
    pageSize: 10,
    orderBy: 'auditDate desc',
    totalCount: 0,
    auditTemplateName: '',
    auditorName: '',
    title: '',
    auditorId: '',
    departmentId: '',
    auditStatus: '',
    auditType: '',
    logStatus: '',
  };
  filterParameter$: Subject<string> = new Subject<string>();
  _auditorByIdFilter = '';
  _departmentIdFilter = '';
  _auditStatusFilter = '';
  _titleFilter = '';
  _logStautsFilter = '';

  @ViewChild(MatPaginator) paginator: MatPaginator;
  @ViewChild(MatSort) sort: MatSort;

  auditStatusList = Object.keys(AUDIT_STATUS)
    .filter((key) => !isNaN(Number(AUDIT_STATUS[key as any])))
    .map((key) => ({
      label: key,
      value: AUDIT_STATUS[key as keyof typeof AUDIT_STATUS],
    }));

  logStatus = Object.keys(AUDIT_LOG_STATUS)
    .filter((key) => !isNaN(Number(AUDIT_LOG_STATUS[key as any]))) // filter numeric values
    .map((key) => ({
      id: key,
      name: AUDIT_LOG_STATUS[key as keyof typeof AUDIT_LOG_STATUS],
    }));

  public get titleFilter(): string {
    return this._titleFilter;
  }

  public set titleFilter(v: string) {
    if (this._titleFilter !== v) {
      this._titleFilter = v;
      const titleFilter = `title#${v}`;
      this.filterParameter$.next(titleFilter);
    }
  }

  public get auditorIdFilter(): string {
    return this._auditorByIdFilter;
  }

  public set auditorIdFilter(v: string) {
    if (this._auditorByIdFilter !== v) {
      this._auditorByIdFilter = v;
      const auditorByIdFilter = `auditorName#${v}`;
      this.filterParameter$.next(auditorByIdFilter);
    }
  }

  public get departmentIdFilter(): string {
    return this._departmentIdFilter;
  }

  public set departmentIdFilter(v: string) {
    if (this._departmentIdFilter !== v) {
      this._departmentIdFilter = v;
      const departmentIdFilter = `departmentId#${v}`;
      this.filterParameter$.next(departmentIdFilter);
    }
  }

  public get auditStatusFilter(): string {
    return this._auditStatusFilter;
  }

  public set auditStatusFilter(v: string) {
    if (this._auditStatusFilter !== v) {
      this._auditStatusFilter = v;
      const auditStatusFilter = `auditStatus#${v}`;
      this.filterParameter$.next(auditStatusFilter);
    }
  }

  public get logStautsFilter(): string {
    return this._logStautsFilter;
  }

  public set logStautsFilter(v: string) {
    if (this._logStautsFilter !== v) {
      this._logStautsFilter = v;
      const logStatusFilter = `logStatus#${v}`;
      this.filterParameter$.next(logStatusFilter);
    }
  }

  reFresh() {
    this.loadByQuery(this.auditFilterParameter);
  }

  ngOnInit(): void {
    this.reFresh();

    this.sub$.sink = this.filterParameter$
      .pipe(debounceTime(1000), distinctUntilChanged())
      .subscribe((c: string) => {
        this.auditFilterParameter.skip = 0;
        this.paginator.pageIndex = 0;
        const filterArray: Array<string> = c.split('#');
        if (filterArray[0] === 'title') {
          this.auditFilterParameter.title = filterArray[1];
        } else if (filterArray[0] === 'auditorName') {
          this.auditFilterParameter.auditorId = filterArray[1];
        } else if (filterArray[0] === 'auditStatus') {
          this.auditFilterParameter.auditStatus = filterArray[1];
        } else if (filterArray[0] === 'departmentId') {
          this.auditFilterParameter.departmentId = filterArray[1];
        } else if (filterArray[0] === 'logStatus') {
          this.auditFilterParameter.logStatus = filterArray[1];
        }
        this.reFresh();
      });
  }

  ngAfterViewInit(): void {
    this.sub$.sink = this.sort.sortChange.subscribe(() => {
      this.paginator.pageIndex = 0;
    });
    this.sub$.sink = merge(this.sort.sortChange, this.paginator.page)
      .pipe(
        tap(() => {
          this.auditFilterParameter.skip =
            this.paginator.pageIndex * this.paginator.pageSize;
          this.auditFilterParameter.pageSize = this.paginator.pageSize;
          this.auditFilterParameter.orderBy =
            this.sort.active + ' ' + this.sort.direction;
          this.reFresh();
        })
      )
      .subscribe();
  }

  loadByQuery(filter: AuditResource) {
    this.sub$.sink = this.auditLogService
      .getAuditResponseLogs(filter)
      .subscribe({
        next: (httpResponse: HttpResponse<AuditResponseLog[]>) => {
          if (httpResponse && httpResponse.body) {
            this.auditLogList = [...(httpResponse.body as Audit[])];
          }
          const paginationHeader = httpResponse.headers.get('x-pagination');
          if (paginationHeader) {
            const pagination = JSON.parse(paginationHeader);
            this.auditFilterParameter = {
              ...filter,
              totalCount: pagination.totalCount,
            };
          }
        },
      });
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.auditLogList.indexOf(row);
  }
}
