import { Component, ElementRef, EventEmitter, inject, Input, OnInit, Output, QueryList, ViewChildren } from '@angular/core';
import jsPDF from 'jspdf';
import html2canvas from 'html2canvas';
import { DatePipe, DecimalPipe, NgClass, NgStyle } from '@angular/common';
import { ActivatedRoute, RouterModule } from '@angular/router';
import { BaseComponent } from '../../base.component';
import { AuditService } from '../audit.service';
import { Audit } from '@core/domain-classes/audit';
import { AuditTypePipe } from "../audit-type.pipe";
import { MatCheckboxModule } from '@angular/material/checkbox';
import { AuditTemplate } from '../../audit-template/models/audit-template';
import { MatRadioModule } from '@angular/material/radio';
import { MatIconModule } from '@angular/material/icon';
import { AUDIT_STATUS } from '../../audit-template/models/audit-status';
import { SecurityService } from '@core/security/security.service';
import { AuditReviewComponent } from '../audit-review/audit-review.component';
import { MatDialog } from '@angular/material/dialog';
import { MatButtonModule } from '@angular/material/button';
import { NON_CONFORMANCE_STATUS } from '../../non-conformance/model/non-conformance-status-enum';
import { NON_CONFORMANCE_SEVERITY } from '../../non-conformance/model/non-conformance-severity-enum';
import { CAPA_ACTION_TYPE } from '../../capa/model/capa-action-type-enum';
import { CAPA_ACTION_STATUS } from '../../capa/model/capa-action-status-enum';
import { CAPA_REVIEW_STATUS } from '../../capa/model/capa-review-status-enum';
import { TranslateModule } from '@ngx-translate/core';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { AuditResponseAttachment } from '../models/audit-response-attachments';
import { DocumentView } from '@core/domain-classes/document-view';
import { QmsModuleEnum } from '@core/domain-classes/qms-module-enum';
import { BasePreviewComponent } from '@shared/base-preview/base-preview.component';
import { OverlayPanel } from '@shared/overlay-panel/overlay-panel.service';
import { QUESTION_TYPE } from '../../audit-template/models/question-type';
import { AuditStatusPipe } from '../audit-status.pipe';
import { CAPA_REQUEST_STATUS } from '../../capa/model/capa-Request-Status-enum';
import { DownloadAuditReport } from "./download-audit-report/download-audit-report";
import { ToastrService } from '@core/services/toastr-service';
import { MatProgressSpinner } from "@angular/material/progress-spinner";

@Component({
  selector: 'app-audit-report',
  imports: [
    AuditTypePipe,
    RouterModule,
    MatCheckboxModule,
    DecimalPipe,
    MatButtonModule,
    MatRadioModule,
    MatIconModule,
    TranslateModule,
    UTCToLocalTime,
    HasClaimDirective,
    AuditStatusPipe,
    DatePipe,
    NgClass,
    NgStyle,
    DownloadAuditReport,
    MatProgressSpinner
],
  templateUrl: './audit-report.component.html',
  styleUrl: './audit-report.component.scss'
})
export class AuditReportComponent extends BaseComponent implements OnInit {
  @Input() auditId: string = '';
  @Input() isDownload: boolean = false;
  @Output() downloadComplete = new EventEmitter<void>();
  route = inject(ActivatedRoute);
  auditService = inject(AuditService);
  securityService = inject(SecurityService);
  dialog = inject(MatDialog);
  toastrService = inject(ToastrService);
  AUDIT_STATUS = AUDIT_STATUS;
  Non_Conformance_Status = NON_CONFORMANCE_STATUS;
  Non_Conformance_Severity = NON_CONFORMANCE_SEVERITY;
  CAPA_ACTION_TYPE = CAPA_ACTION_TYPE;
  CAPA_ACTION_STATUS = CAPA_ACTION_STATUS;
  CAPA_REVIEW_STATUS = CAPA_REVIEW_STATUS;
  CAPA_REQUEST_STATUS = CAPA_REQUEST_STATUS;
  overlay = inject(OverlayPanel);
  auditDetails: Audit = {
    id: '',
    auditorId: '',
    title: '',
    auditTemplateId: '',
    auditTemplate: {} as AuditTemplate,
    departmentName: '',
    auditorName: '',
    status: undefined,
    totalScore: undefined,
    effectiveScore: undefined,
    capaRequests: [],
    nonConformances: [],
    auditNumber: '',
    auditQuestionResponses: [],
  };
  QUESTION_TYPE = QUESTION_TYPE;

  ngOnInit(): void {
    this.sub$.sink = this.route.paramMap.subscribe((data) => {
      const auditId = data.get('id');
      if (auditId) {
        this.getAuditDetails(auditId);
      } else if (this.auditId) {
        this.getAuditDetails(this.auditId);
      }
    });
  }

  generateAuditPDF(): void {
    this.isDownload = true;
  }

  completeDownload(event: any) {
    this.isDownload = false;

    this.toastrService.success(this.translationService.getValue('AUDIT_REPORT_DOWNLOADED_SUCCESSFULLY'));
    if (this.auditId) {
      this.downloadComplete.emit();
    }
  }

  getFileIcon(fileType: string): string {
    if (fileType.startsWith('image/')) {
      return 'image';
    } else if (fileType.includes('pdf')) {
      return 'picture_as_pdf';
    } else if (fileType.includes('spreadsheet') || fileType.includes('excel')) {
      return 'table_chart';
    } else if (fileType.includes('document') || fileType.includes('word')) {
      return 'description';
    } else {
      return 'insert_drive_file';
    }
  }

  onMediaView(document: AuditResponseAttachment) {
    const documentView: DocumentView = {
      documentId: document.id,
      name: document.fileName,
      extension: document.extension,
      isVersion: false,
      isFromPublicPreview: false,
      isPreviewDownloadEnabled: true,
      isFileRequestDocument: false,
      totalChunk: document.totalChunk,
      moduleNo: QmsModuleEnum.Audit,
    };
    this.overlay.open(BasePreviewComponent, {
      position: 'center',
      origin: 'global',
      panelClass: ['file-preview-overlay-container', 'white-background'],
      data: documentView,
    });
  }

  getCountArray(count: any): number[] {
    const n = Number(count);
    // Check if n is a valid, finite, non-negative integer
    if (!Number.isFinite(n) || n < 0) {
      return []; // fallback to safe empty array
    }
    return Array.from({ length: Math.floor(n) }, (_, i) => i);
  }

  getAuditDetails(id: string) {
    this.sub$.sink = this.auditService.getAuditReportById(id).subscribe({
      next: (audit: Audit) => {
        if (audit) {
          this.auditDetails = audit;
          if (this.auditId) {
            this.generateAuditPDF();
          }
        }
      },
    })
  }

  submitAudit(status: AUDIT_STATUS): void {
    this.dialog.open(AuditReviewComponent, {
      data: { auditId: this.auditDetails?.id, status: status },
    });
  }
}
