import { DecimalPipe, DatePipe, NgClass, NgStyle } from '@angular/common';
import { Component, ElementRef, EventEmitter, inject, Input, OnChanges, OnInit, Output, QueryList, ViewChildren } from '@angular/core';
import { MatButtonModule } from '@angular/material/button';
import { MatCheckboxModule } from '@angular/material/checkbox';
import { MatIconModule } from '@angular/material/icon';
import { MatRadioModule } from '@angular/material/radio';
import { ActivatedRoute, RouterModule } from '@angular/router';
import { TranslateModule } from '@ngx-translate/core';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { AuditStatusPipe } from '../../audit-status.pipe';
import { AuditTypePipe } from '../../audit-type.pipe';
import { AUDIT_STATUS } from '../../../audit-template/models/audit-status';
import { AuditReviewComponent } from '../../audit-review/audit-review.component';
import { MatDialog } from '@angular/material/dialog';
import { Audit } from '@core/domain-classes/audit';
import { SecurityService } from '@core/security/security.service';
import { OverlayPanel } from '@shared/overlay-panel/overlay-panel.service';
import html2canvas from 'html2canvas';
import jsPDF from 'jspdf';
import { QUESTION_TYPE } from '../../../audit-template/models/question-type';
import { CAPA_ACTION_STATUS } from '../../../capa/model/capa-action-status-enum';
import { CAPA_ACTION_TYPE } from '../../../capa/model/capa-action-type-enum';
import { CAPA_REQUEST_STATUS } from '../../../capa/model/capa-Request-Status-enum';
import { CAPA_REVIEW_STATUS } from '../../../capa/model/capa-review-status-enum';
import { NON_CONFORMANCE_SEVERITY } from '../../../non-conformance/model/non-conformance-severity-enum';
import { NON_CONFORMANCE_STATUS } from '../../../non-conformance/model/non-conformance-status-enum';
import { AuditService } from '../../audit.service';
import { BaseComponent } from '../../../base.component';

@Component({
  selector: 'app-download-audit-report',
  imports: [
    AuditTypePipe,
    RouterModule,
    MatCheckboxModule,
    DecimalPipe,
    MatButtonModule,
    MatRadioModule,
    MatIconModule,
    TranslateModule,
    UTCToLocalTime,
    HasClaimDirective,
    AuditStatusPipe,
    DatePipe,
    NgClass,
    NgStyle
  ],
  templateUrl: './download-audit-report.html',
  styleUrl: './download-audit-report.scss'
})
export class DownloadAuditReport extends BaseComponent implements OnInit, OnChanges {
  @Input() auditDetails: Audit | null = null;
  @Input() isDownload: boolean = false;
  @Output() downloadComplete = new EventEmitter<void>();
  route = inject(ActivatedRoute);
  auditService = inject(AuditService);
  securityService = inject(SecurityService);
  dialog = inject(MatDialog);
  AUDIT_STATUS = AUDIT_STATUS;
  Non_Conformance_Status = NON_CONFORMANCE_STATUS;
  Non_Conformance_Severity = NON_CONFORMANCE_SEVERITY;
  CAPA_ACTION_TYPE = CAPA_ACTION_TYPE;
  CAPA_ACTION_STATUS = CAPA_ACTION_STATUS;
  CAPA_REVIEW_STATUS = CAPA_REVIEW_STATUS;
  CAPA_REQUEST_STATUS = CAPA_REQUEST_STATUS;
  overlay = inject(OverlayPanel);
  QUESTION_TYPE = QUESTION_TYPE;
  auditDetailsReport = {} as Audit;

  ngOnInit(): void {
  }

  ngOnChanges(changes: any): void {
    if (changes['auditDetails'] && this.auditDetails) {
      this.auditDetailsReport = this.auditDetails;
    }
    if (changes['isDownload'] && this.isDownload) {
      this.generateAuditPDF();
    }
  }

  @ViewChildren('section,section1, section2, section3, section4, section5') sectionRefs!: QueryList<ElementRef>;

  generateAuditPDF(): void {
    const pdf = new jsPDF('p', 'mm', 'a4');
    const pdfWidth = pdf.internal.pageSize.getWidth();
    const pdfHeight = pdf.internal.pageSize.getHeight();
    let cursorY = 10;

    const processSections = async () => {
      const elements = this.sectionRefs.toArray();

      for (let i = 0; i < elements.length; i++) {
        const el = elements[i].nativeElement;

        const canvas = await html2canvas(el, {
          scale: 2, // 🔻 reduce scale to lower resolution
          scrollY: -window.scrollY,
          useCORS: true,
        });

        // 📸 Convert canvas to compressed JPEG
        const imgData = canvas.toDataURL('image/jpeg', 0.85); // 70% quality

        const imgWidth = pdfWidth - 20; // 10mm margin on each side
        const ratio = canvas.height / canvas.width;
        const imgHeight = imgWidth * ratio;

        if (cursorY + imgHeight > pdfHeight - 10) {
          pdf.addPage();
          cursorY = 10;
        }

        pdf.addImage(imgData, 'JPEG', 10, cursorY, imgWidth, imgHeight);
        cursorY += imgHeight + 5; // Spacing between sections
      }

      pdf.save('Audit_Report.pdf');

      // Notify parent component that download is complete
      this.downloadComplete.emit();
    };

    setTimeout(() => processSections(), 500); // Wait for view to render
  }

  getFileIcon(fileType: string): string {
    if (fileType.startsWith('image/')) {
      return 'image';
    } else if (fileType.includes('pdf')) {
      return 'picture_as_pdf';
    } else if (fileType.includes('spreadsheet') || fileType.includes('excel')) {
      return 'table_chart';
    } else if (fileType.includes('document') || fileType.includes('word')) {
      return 'description';
    } else {
      return 'insert_drive_file';
    }
  }

  getCountArray(count: any): number[] {
    const n = Number(count);
    // Check if n is a valid, finite, non-negative integer
    if (!Number.isFinite(n) || n < 0) {
      return []; // fallback to safe empty array
    }
    return Array.from({ length: Math.floor(n) }, (_, i) => i);
  }

  submitAudit(status: AUDIT_STATUS): void {
    this.dialog.open(AuditReviewComponent, {
      data: { auditId: this.auditDetails?.id, status: status },
    });
  }
}
