import { Component, Inject, inject, OnInit } from '@angular/core';
import { FormBuilder, FormGroup, ReactiveFormsModule, Validators } from '@angular/forms';
import { MatButtonModule } from '@angular/material/button';
import { MAT_DIALOG_DATA, MatDialogModule, MatDialogRef } from '@angular/material/dialog';

import { ToastrService } from '@core/services/toastr-service';
import { BaseComponent } from '../../base.component';
import { UserStore } from '../../user/store/user.store';
import { Router } from '@angular/router';
import { AuditReview } from '@core/domain-classes/audit';
import { AUDIT_STATUS } from '../../audit-template/models/audit-status';
import { AuditReviewService } from './audit-review.service';
import { AuditStore } from '../audit-store';
import { AuditReviewStore } from '../audit-for-review-list/audit-review-store';
import { TranslateModule } from '@ngx-translate/core';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';

@Component({
  selector: 'app-audit-review',
  imports: [
    MatDialogModule,
    ReactiveFormsModule,
    MatButtonModule,
    TranslateModule,
    PageHelpTextComponent,
    MatIconModule,
    MatCardModule
  ],
  templateUrl: './audit-review.component.html',
  styleUrl: './audit-review.component.scss'
})
export class AuditReviewComponent extends BaseComponent implements OnInit {
  auditReviewForm: FormGroup;
  userStore = inject(UserStore);
  auditStore = inject(AuditStore);
  auditReviewStore = inject(AuditReviewStore);
  auditFilterParameter = { ...this.auditStore.filterParameters() };
  auditReviewFilterParameter = { ...this.auditReviewStore.filterParameters() };
  constructor(
    private fb: FormBuilder,
    private toastrService: ToastrService,
    private auditReviewService: AuditReviewService,
    private dialogRef: MatDialogRef<AuditReviewComponent>,
    @Inject(MAT_DIALOG_DATA) public data: { auditId: string, status?: AUDIT_STATUS },
    private router: Router
  ) {
    super();
    this.data = { auditId: data.auditId, status: data.status };
  }
  ngOnInit(): void {
    this.createForm();
  }

  createForm(): void {
    const isCommentRequired = this.data.status === AUDIT_STATUS.REJECTED || this.data.status === AUDIT_STATUS.REWORK;
    this.auditReviewForm = this.fb.group({
      auditId: [this.data.auditId],
      comment: ['', isCommentRequired ? Validators.required : []],
    });
  }

  saveAuditReview(): void {
    if (this.auditReviewForm.invalid) {
      this.auditReviewForm.markAllAsTouched();
      return;
    }
    const auditReview: AuditReview = {
      auditId: this.data.auditId,
      status: this.data.status,
      comments: this.auditReviewForm.get('comment')?.value,
    };
    this.auditReviewService.submitAuditReview(auditReview).subscribe({
      next: (response) => {
        this.toastrService.success(
          this.translationService.getValue('AUDIT_REVIEW_SUCCESSFULLY')
        );
        this.dialogRef.close();
        this.auditStore.loadByQuery(this.auditFilterParameter);
        this.auditReviewStore.loadByQuery(this.auditReviewFilterParameter);
        this.router.navigate(['/audit/audit-list']);
      }
    });
  }

  closeDialog() {
    this.dialogRef.close();
  }
}
