import {
  patchState,
  signalStore,
  withHooks,
  withMethods,
  withState,
} from '@ngrx/signals';
import { rxMethod } from '@ngrx/signals/rxjs-interop';
import { inject } from '@angular/core';

import { ToastrService } from '@core/services/toastr-service';
import { TranslationService } from '@core/services/translation.service';
import { distinctUntilChanged, pipe, switchMap, tap } from 'rxjs';
import { tapResponse } from '@ngrx/operators';
import { toObservable } from '@angular/core/rxjs-interop';
import { HttpResponse } from '@angular/common/http';
import { Audit, StartAudit } from '@core/domain-classes/audit';
import { AuditService } from './audit.service';
import { AuditResource } from './audit-resource';
import { Router } from '@angular/router';

type AuditState = {
  auditList: Audit[];
  audit: Audit;
  loadList: boolean;
  isAddUpdate: boolean;
  filterParameters: AuditResource;
};

export const initialAuditState: AuditState = {
  auditList: [],
  audit: {} as Audit,
  loadList: true,
  isAddUpdate: false,
  filterParameters: {
    orderBy: 'AuditTemplateName asc',
    pageSize: 10,
    skip: 0,
    totalCount: 0,
    auditTemplateName: '',
    title: '',
    auditStatus: '',
    auditorName: '',
    auditType: '',
    departmentId: '',
    auditorId: '',
  },
};

export const AuditStore = signalStore(
  { providedIn: 'root' },
  withState(initialAuditState),
  withMethods(
    (
      store,
      auditService = inject(AuditService),
      router = inject(Router),
      toastrService = inject(ToastrService),
      translationService = inject(TranslationService)
    ) => ({
      loadByQuery: rxMethod<AuditResource>(
        pipe(
          tap(() => patchState(store, { auditList: [] })),
          switchMap((filter) => {
            return auditService.getAudits(filter).pipe(
              tapResponse({
                next: (httpResponse: HttpResponse<Audit[]>) => {
                  if (httpResponse && httpResponse.body) {
                    patchState(store, {
                      auditList: [...httpResponse.body],
                      loadList: false,
                    });
                  }
                  if (httpResponse.headers.get('x-pagination')) {
                    const pagination: AuditResource = JSON.parse(
                      httpResponse.headers.get('x-pagination') ?? ''
                    );
                    patchState(store, {
                      filterParameters: {
                        ...filter,
                        totalCount: pagination.totalCount,
                      },
                    });
                  }
                },
                error: (error: any) => {
                  toastrService.error(error.error);
                },
              })
            );
          })
        )
      ),
      addAudit: rxMethod<StartAudit>(
        pipe(
          distinctUntilChanged(),
          tap(() => patchState(store, { loadList: false })),
          switchMap((startAudit: StartAudit) =>
            auditService.startAudit(startAudit).pipe(
              tapResponse({
                next: (addAudit: StartAudit) => {
                  toastrService.success(
                    translationService.getValue('START_AUDIT_SUCCESSFULLY')
                  );
                  router.navigate(['/audit', addAudit.id]);
                  patchState(store, { isAddUpdate: true, loadList: true });
                },
                error: (err: any) => {
                  toastrService.error(
                    err?.error?.message ||
                    translationService.getValue('ERROR_OCCURRED')
                  );
                  patchState(store, { loadList: true });
                },
              })
            )
          )
        )
      ),
      updateAudit: rxMethod<StartAudit>(
        pipe(
          distinctUntilChanged(),
          tap(() => patchState(store, { loadList: false })),
          switchMap((updateAudit: StartAudit) =>
            auditService.updateAudit(updateAudit).pipe(
              tapResponse({
                next: (response: StartAudit) => {
                  patchState(store, {
                    auditList: store.auditList().map((cr) => {
                      if (response.id === cr.id) {
                        return {
                          ...cr,
                          status: Number(response.status ?? cr.status),
                        };
                      }
                      return cr;
                    }),
                    isAddUpdate: true,
                  });
                },
                error: (err: any) => {
                  toastrService.error(
                    err?.error?.message ||
                    translationService.getValue('ERROR_OCCURRED')
                  );
                  patchState(store, { loadList: true });
                },
              })
            )
          )
        )
      ),
      resetflag() {
        patchState(store, { isAddUpdate: false });
      },
      setloadListFlag(flag: boolean) {
        patchState(store, { loadList: flag });
      }
    })
  ),
  withHooks({
    onInit(store) {
      toObservable(store.loadList).subscribe((flag) => {
        if (flag) {
          store.loadByQuery(store.filterParameters());
        }
      });
    },
  })
);
