import { NgClass } from '@angular/common';
import { Component, inject, OnDestroy, OnInit } from '@angular/core';
import { ReactiveFormsModule } from '@angular/forms';
import { ActivatedRoute, RouterModule } from '@angular/router';
import { MultiSelectQuestionComponent } from './question-types/multi-select-question/multi-select-question.component';
import { FileUploadQuestionComponent } from './question-types/file-upload-question/file-upload-question.component';
import { YesNoQuestionComponent } from './question-types/yes-no-question/yes-no-question.component';
import { RatingQuestionComponent } from './question-types/rating-question/rating-question.component';
import { MatStepperModule } from '@angular/material/stepper';
import { MatButtonModule } from '@angular/material/button';
import { MatIconModule } from '@angular/material/icon';
import { MatProgressBarModule } from '@angular/material/progress-bar';
import { MatDialog, MatDialogModule } from '@angular/material/dialog';
import { MatCardModule } from '@angular/material/card';
import { AuditTemplate } from '../audit-template/models/audit-template';
import { BehaviorSubject, Subscription, switchMap } from 'rxjs';
import { AuditTemplateQuestion } from '../audit-template/models/audit-template-question';
import { TranslateModule } from '@ngx-translate/core';
import { AuditTypePipe } from './audit-type.pipe';
import { QUESTION_TYPE } from '../audit-template/models/question-type';
import { InputQuestionComponent } from './question-types/input-question/input-question.component';
import { NumberQuestionComponent } from './question-types/number-question/number-question.component';
import { DateQuestionComponent } from './question-types/date-question/date-question.component';
import { AuditResponse } from './models/audit-response';
import { Audit } from '@core/domain-classes/audit';

import { ToastrService } from '@core/services/toastr-service';
import { TranslationService } from '@core/services/translation.service';
import { AUDIT_STATUS } from '../audit-template/models/audit-status';
import { AuditStore } from './audit-store';
import { AuditService } from './audit.service';
import { CAPA_REQUEST_STATUS } from '../capa/model/capa-Request-Status-enum';
import { SubmitAuditComponent } from './submit-audit/submit-audit.component';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { AuditStatusPipe } from './audit-status.pipe';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { AuditReviewer } from '@core/domain-classes/audit-reviewer';
import { MatTableModule } from '@angular/material/table';
import { NonConformance } from '../non-conformance/model/non-conformance';
import { CapaRequest } from '../capa/model/capa-request';
import { CapaRequestStatusPipe } from "../capa/pipes/capa-request-status.pipe";
import { NonConformanceStatusPipe } from "../non-conformance/pipes/non-conformance-status.pipe";
import { ManageCapaRequestComponent } from '../capa/capa-request/manage-capa-request/manage-capa-request.component';
import { ManageNonConformanceComponent } from '../non-conformance/manage-non-conformance/manage-non-conformance.component';

@Component({
  selector: 'app-audit',
  standalone: true,
  imports: [
    RouterModule,
    ReactiveFormsModule,
    MultiSelectQuestionComponent,
    DateQuestionComponent,
    FileUploadQuestionComponent,
    YesNoQuestionComponent,
    InputQuestionComponent,
    NumberQuestionComponent,
    RatingQuestionComponent,
    MatStepperModule,
    MatButtonModule,
    MatIconModule,
    MatProgressBarModule,
    MatDialogModule,
    MatCardModule,
    MatTableModule,
    TranslateModule,
    AuditTypePipe,
    HasClaimDirective,
    AuditStatusPipe,
    UTCToLocalTime,
    MatCardModule,
    NgClass,
    CapaRequestStatusPipe,
    NonConformanceStatusPipe
  ],
  templateUrl: './audit.component.html',
  styleUrl: './audit.component.scss',
})
export class AuditComponent implements OnInit, OnDestroy {
  templateId: string | null = null;
  template: AuditTemplate | null = null;
  audit: Audit | null = null;
  AUDIT_STATUS = AUDIT_STATUS;
  CAPA_REQUEST_STATUS = CAPA_REQUEST_STATUS;
  questions: AuditTemplateQuestion[] = [];
  loading = true;
  currentProgress = 0;
  currentQuestionIndex = 0;
  answeredQuestions = new Set<string>();
  activeQuestionIndex: number;
  response: AuditResponse[] = [];
  QUESTION_TYPE = QUESTION_TYPE;
  questionVisibility: { [id: string]: boolean } = {};
  private subscriptions = new Subscription();
  auditStore = inject(AuditStore);
  dialog = inject(MatDialog);
  auditFilterParameter = { ...this.auditStore.filterParameters() };
  auditReviewers: AuditReviewer[] = [];
  ncDisplayedColumns: string[] = ['ncNumber', 'title', 'status'];
  capaDisplayedColumns: string[] = ['referenceNo', 'title', 'status'];
  ncDataSource: NonConformance[] = [];
  capaDataSource: CapaRequest[] = [];

  constructor(
    private route: ActivatedRoute,
    private auditService: AuditService,
    private toastrService: ToastrService,
    private translationService: TranslationService,
  ) {
  }

  ngOnInit(): void {
    this.loadAuditTemplate();
    // Set up scroll detection for tracking current question
    window.addEventListener('scroll', this.onScroll.bind(this));
    //this.initializeQuestionVisibility();
  }


  ngOnDestroy(): void {
    this.subscriptions.unsubscribe();
    window.removeEventListener('scroll', this.onScroll.bind(this));
  }

  markQuestionAsCompleted(questionId: string): void {
    this.answeredQuestions.add(questionId);
    this.updateProgress();
  }

  updateProgress(): void {
    const total = this.audit?.auditTemplate.questions.length ?? 0;
    const answered = this.answeredQuestions.size;
    this.currentProgress = total > 0 ? Math.round((answered / total) * 100) : 0;

  }

  private initializeQuestionVisibility(): void {
    if (this.template?.questions) {
      this.template.questions.forEach((question) => {
        this.questionVisibility[question.id ?? ''] = false;
      });
    }
  }

  onResponseChanged(response: AuditResponse): void {
    const existing = this.response.find(
      (a) => a.questionId === response.questionId
    );
    if (existing) {
      existing.response = response.response;
    } else {
      this.response.push(response);
      this.answeredQuestions.add(response.questionId);
      this.updateProgress();
    }
  }

  toggleQuestionVisibility(id: string): void {
    this.questionVisibility[id] = !this.questionVisibility[id];
  }

  private loadAuditTemplate(): void {
    this.loading = true;
    const sub = this.route.paramMap
      .pipe(
        switchMap((params) => {
          this.templateId = params.get('id');
          if (!this.templateId) {
            throw new Error('Template ID is required');
          }
          return this.auditService.getAuditById(this.templateId);
        })
      )
      .subscribe({
        next: (audit: Audit) => {
          this.audit = audit;
          this.getNonConformanceByAuditId(this.audit?.id ?? '');
          this.getCapaByAuditId(this.audit?.id ?? '');

          if (Array.isArray(audit.auditReviewers) && audit.auditReviewers.length > 0) {
            this.auditReviewers = [...audit.auditReviewers];
          }
          this.questions = audit?.auditTemplate?.questions || [];
          this.initializeQuestionVisibility();
          this.loading = false;
        },
        error: (error) => {
          this.loading = false;
          this.toastrService.error(
            this.translationService.getValue('ERROR_LOADING_AUDIT_TEMPLATE') +
            error.message
          );
        },
      });
    this.subscriptions.add(sub);
  }

  getNonConformanceByAuditId(auditId: string): void {
    this.auditService.getNonConformanceByAuditId(auditId).subscribe({
      next: (nc: NonConformance[]) => {
        this.ncDataSource = nc as NonConformance[];
      }
    });
  }

  getCapaByAuditId(auditId: string): void {
    this.auditService.getCapaByAuditId(auditId).subscribe({
      next: (capa: CapaRequest[]) => {
        this.capaDataSource = capa as CapaRequest[];
      }
    });
  }

  getAuditReviewersById(): void {
    const sub = this.auditService.getAuditReviewersById(this.audit?.id ?? '').subscribe({
      next: (audit: AuditReviewer[]) => {
        this.auditReviewers = audit;
      },
      error: (error) => {
        this.toastrService.error(
          this.translationService.getValue('ERROR_LOADING_AUDIT_TEMPLATE') +
          error.message
        );
      },
    });
    this.subscriptions.add(sub);
  }

  scrollToQuestion(index: number): void {
    this.activeQuestionIndex = index;
    const element = document.getElementById(`question-${index}`);
    if (element) {
      element.scrollIntoView({ behavior: 'smooth' });
    }
  }

  private onScroll(): void {
    // Determine which question is currently in view
    if (this.audit && this.audit.auditTemplate && this.audit.auditTemplate.questions) {
      for (let i = 0; i < this.audit.auditTemplate.questions.length; i++) {
        const element = document.getElementById(`question-${i}`);

        if (element) {
          const rect = element.getBoundingClientRect();
          const isInView = rect.top >= 0 && rect.top <= window.innerHeight / 2;

          if (isInView) {
            this.currentQuestionIndex = i;
            break;
          }
        }
      }
    }
  }

  isQuestionAnswered(questionId: string): boolean {
    return this.answeredQuestions.has(questionId);
  }

  submitAudit(): void {
    try {
      const questions = this.audit?.auditTemplate?.questions || [];
      const unanswered = questions.filter(
        (q) => !this.response.find((r) => r.questionId === q.id)
      );
      if (unanswered.length > 0) {
        this.toastrService.error(
          this.translationService.getValue(
            'SOME_ANSWERS_ARE_MISSING_PLEASE_ANSWER_ALL_QUESTIONS_BEFORE_SUBMITTING'
          )
        );
        return;
      }

      const nonConformanceExists = (this.audit?.nonConformances || []).some(
        (nc) => nc?.status === 0 || nc?.status === 1
      );
      if (nonConformanceExists) {
        this.toastrService.error(
          this.translationService.getValue(
            'NON_CONFORMANCE_SUBMIT_BEFORE_AUDIT'
          )
        );
        return;
      }

      const capaRequestExists = (this.audit?.capaRequests || []).some(
        (capa) =>
          capa?.capaRequestStatus == 1 || capa?.capaRequestStatus == 2
      );
      if (capaRequestExists) {
        this.toastrService.error(
          this.translationService.getValue('CAPA_REQUEST_SUBMIT_BEFORE_AUDIT')
        );
        return;
      }
      this.dialog.open(SubmitAuditComponent, {
        data: { auditId: this.audit?.id },
      });
    } catch (error: any) {
      this.toastrService.error(
        this.translationService.getValue('ERROR_SUBMITTING_AUDIT') +
        (error?.message ? ': ' + error.message : '')
      );
    }
  }

  isOddDataRow(index: number): boolean {
    return index % 2 !== 0;
  }

  getDataIndex(row: any): number {
    return this.ncDataSource.indexOf(row);
  }

  getDataIndexCapa(row: any): number {
    return this.capaDataSource.indexOf(row);
  }

  openCapaRequestDialog(event: { auditId: string; auditResponseId: string }): void {
    const dialogRef = this.dialog.open(ManageCapaRequestComponent, {
      width: '100%',
      maxWidth: '60vw',
      maxHeight: '90vh',
      data: Object.assign({ event }),
    });

    dialogRef.afterClosed().subscribe((result) => {
      if (result) {
        this.getCapaByAuditId(this.audit?.id ?? '');
        this.auditService.triggerUpdate();
      }
    });
  }

  openNCDialog(event: { auditId: string; auditResponseId: string }): void {
    const dialogRef = this.dialog.open(ManageNonConformanceComponent, {
      width: '100%',
      maxWidth: '60vw',
      maxHeight: '90vh',
      data: Object.assign({ event }),
    });

    dialogRef.afterClosed().subscribe((result) => {
      if (result) {
        this.getNonConformanceByAuditId(this.audit?.id ?? '');
        this.auditService.triggerUpdate();
      }
    });
  }

}
