import { Injectable } from '@angular/core';
import { HttpClient, HttpParams, HttpResponse } from '@angular/common/http';
import { AuditResponse } from './models/audit-response';
import { AuditResponseAttachment } from './models/audit-response-attachments';
import { BehaviorSubject, catchError, Observable, retry, timer } from 'rxjs';
import { AuditResponseAttachmentChunk } from './models/audit-response-attachment-chunk';
import { CommonError } from '@core/error-handler/common-error';
import { Audit, StartAudit } from '@core/domain-classes/audit';
import { CommonHttpErrorService } from '@core/error-handler/common-http-error.service';
import { AuditQuestionResponse } from '@core/domain-classes/audit-question-respone';
import { AuditResource } from './audit-resource';
import { IdName } from '@core/domain-classes/id-name';
import { AuditReviewer } from '@core/domain-classes/audit-reviewer';
import { AuditNotification } from '@core/domain-classes/audit-notification';
import { NonConformance } from '../non-conformance/model/non-conformance';
import { CapaRequest } from '../capa/model/capa-request';

@Injectable({ providedIn: 'root' })
export class AuditService {
  maxRetries: number = 2;
  constructor(
    private commonHttpErrorService: CommonHttpErrorService,
    private httpClient: HttpClient
  ) { }

  updateTrigger: BehaviorSubject<boolean> = new BehaviorSubject<boolean>(false);

  triggerUpdate() {
    this.updateTrigger.next(true);
  }

  getResponseForQuestion(auditId: string, questionId: string) {
    const url = `audit/${auditId}/question/${questionId}/response`;
    return this.httpClient.get<AuditResponse>(url);
  }

  getResponseWithAttachmentForQuestion(
    questionId: string
  ): Observable<AuditResponse> {
    return this.httpClient.get<AuditResponse>(
      `audit/question/${questionId}/response/attachment`
    );
  }
  saveAuditResponseAttachment(response: AuditResponseAttachment): Observable<AuditResponseAttachment | CommonError> {
    const url = `AuditResponseAttachment`;
    return this.httpClient.post<AuditResponseAttachment>(
      url,
      response
    ).pipe(
      catchError(this.commonHttpErrorService.handleError)
    );
  }

  uploadAuditResponseAttachmentChunk(
    auditResponseAttachmentChunk: FormData
  ): Observable<AuditResponseAttachmentChunk | CommonError> {
    const url = `AuditResponseAttachment/Chunk`;
    return this.httpClient
      .post<AuditResponseAttachmentChunk>(url, auditResponseAttachmentChunk)
      .pipe(
        retry({
          count: this.maxRetries, // Retry up to 2 times
          delay: (error, retryCount) => {
            console.warn(`Retrying chunk... Attempt ${retryCount}`);
            return timer(Math.pow(2, retryCount) * 1000); // Exponential backoff: 1s, 2s, 4s...
          },
        })
      );
  }

  markAsAuditResponseAttachmentChunkComplete(
    auditResponseAttachmentChunkId: string,
    status: boolean
  ): Observable<void | CommonError> {
    const url = `AuditResponseAttachment/chunk/uploadStatus`;
    return this.httpClient
      .post<void>(url, {
        auditResponseAttachmentChunkId: auditResponseAttachmentChunkId,
        status: status,
      })
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  startAudit(audit: StartAudit): Observable<StartAudit> {
    const url = `audit`;
    return this.httpClient
      .post<StartAudit>(url, audit);
  }

  updateAudit(audit: StartAudit): Observable<StartAudit> {
    const url = `audit/${audit.id}`;
    return this.httpClient
      .put<StartAudit>(url, audit);
  }

  getAudits(
    resource: AuditResource
  ): Observable<HttpResponse<Audit[]>> {
    const url = 'audit';
    const customParams = new HttpParams()
      .set('OrderBy', resource.orderBy)
      .set('PageSize', resource.pageSize)
      .set('Skip', resource.skip)
      .set('AuditTemplateName', resource.auditTemplateName)
      .set('Title', resource.title)
      .set('AuditorId', resource.auditorId ?? '')
      .set('DepartmentId', resource.departmentId ?? '')
      .set('AuditStatus', resource.auditStatus ?? '')
      .set('AuditType', resource.auditType ?? '');
    return this.httpClient
      .get<Audit[]>(url, {
        params: customParams,
        observe: 'response',
      });

  }

  getAuditForReviews(
    resource: AuditResource
  ): Observable<HttpResponse<Audit[]>> {
    const url = 'audit/audit-for-review';
    const customParams = new HttpParams()
      .set('OrderBy', resource.orderBy)
      .set('PageSize', resource.pageSize)
      .set('Skip', resource.skip)
      .set('AuditTemplateName', resource.auditTemplateName)
      .set('Title', resource.title)
      .set('AuditorId', resource.auditorId ?? '')
      .set('DepartmentId', resource.departmentId ?? '')
      .set('AuditStatus', resource.auditStatus)
      .set('AuditType', resource.auditType);
    return this.httpClient
      .get<Audit[]>(url, {
        params: customParams,
        observe: 'response',
      });

  }

  saveAuditQuestionResponse(
    auditQuestionResponse: AuditQuestionResponse
  ): Observable<AuditQuestionResponse> {
    const url = `auditresponse`;
    return this.httpClient
      .post<AuditQuestionResponse>(url, auditQuestionResponse);

  }

  updateAuditQuestionResponse(
    auditQuestionResponse: AuditQuestionResponse
  ): Observable<AuditQuestionResponse | CommonError> {
    const url = `auditresponse/${auditQuestionResponse.id}`;
    return this.httpClient
      .put<AuditQuestionResponse>(url, auditQuestionResponse)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  deleteAuditResponseAttachment(
    auditResponseAttachmentId: string
  ): Observable<void | CommonError> {
    const url = `AuditResponseAttachment/${auditResponseAttachmentId}`;
    return this.httpClient
      .delete<void>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getAuditById(id: string): Observable<Audit> {
    const url = 'audit/' + id;
    return this.httpClient
      .get<Audit>(url);
  }

  getAuditReportById(id: string): Observable<Audit> {
    const url = 'Audit/' + id + '/report-details';
    return this.httpClient
      .get<Audit>(url);
  }

  getAuditQuestionsById(id: string): Observable<any[] | CommonError> {
    const url = `audit/${id}/questions`;
    return this.httpClient
      .get<any[]>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getAuditsForDropdown(): Observable<IdName> {
    const url = `audit/dropdown`;
    return this.httpClient
      .get<IdName>(url);
  }

  getAuditReviewersById(id: string): Observable<AuditReviewer[]> {
    const url = `audit/${id}/reviewers`;
    return this.httpClient
      .get<AuditReviewer[]>(url);
  }

  getAuditReviewsToNotify(): Observable<AuditNotification[]> {
    const url = `audit/auditReview/notify`;
    return this.httpClient
      .get<AuditNotification[]>(url);
  }

  getNonConformanceByAuditId(auditId: string): Observable<NonConformance[]> {
    const url = `Audit/ncs/${auditId}`;
    return this.httpClient
      .get<NonConformance[]>(url);
  }

  getCapaByAuditId(auditId: string): Observable<CapaRequest[]> {
    const url = `Audit/capas/${auditId}`;
    return this.httpClient
      .get<CapaRequest[]>(url);
  }
}
