import {
  ChangeDetectorRef,
  Component,
  EventEmitter,
  inject,
  Input,
  OnDestroy,
  OnInit,
  Output,
} from '@angular/core';
import {
  FormArray,
  FormBuilder,
  FormGroup,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { MatButtonModule } from '@angular/material/button';
import { MatIconModule } from '@angular/material/icon';
import { MatProgressBarModule } from '@angular/material/progress-bar';
import { AuditService } from '../../audit.service';
import { AuditTemplateQuestion } from '../../../audit-template/models/audit-template-question';
import { CommonService } from '@core/services/common.service';
import { BaseComponent } from '../../../base.component';
import { AllowFileExtension } from '@core/domain-classes/allow-file-extension';
import { TranslateModule } from '@ngx-translate/core';
import { AuditResponse } from '../../models/audit-response';
import { validateFile } from '@core/domain-classes/extension-types';

import { ToastrService } from '@core/services/toastr-service';
import {
  catchError,
  concatMap,
  from,
  mergeMap,
  Observable,
  of,
  toArray,
} from 'rxjs';
import { AuditResponseAttachment } from '../../models/audit-response-attachments';
import { environment } from '@environments/environment';
import { AuditResponseAttachmentChunk } from '../../models/audit-response-attachment-chunk';
import { CommonError } from '@core/error-handler/common-error';
import { DocumentChunkStatus } from '@core/domain-classes/document-chunk-status';
import { AuditQuestionResponse } from '@core/domain-classes/audit-question-respone';
import { AuditResponseChunkStatus } from '../../models/audit-response-chunk-status';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { StartAudit } from '@core/domain-classes/audit';
import { AUDIT_STATUS } from '../../../audit-template/models/audit-status';
import { AuditStore } from '../../audit-store';
import { RouterModule } from '@angular/router';
import { MatTabsModule } from '@angular/material/tabs';
import { TotalCapaListDetailsComponent } from '../../audit-response-details/total-capa-list-details/total-capa-list-details.component';
import { TotalNonConformanceListDetailsComponent } from '../../audit-response-details/total-non-conformance-list-details/total-non-conformance-list-details.component';
import { DocumentView } from '@core/domain-classes/document-view';
import { OverlayPanel } from '@shared/overlay-panel/overlay-panel.service';
import { BasePreviewComponent } from '@shared/base-preview/base-preview.component';
import { QmsModuleEnum } from '@core/domain-classes/qms-module-enum';

@Component({
  selector: 'app-file-upload-question',
  standalone: true,
  imports: [
    ReactiveFormsModule,
    MatButtonModule,
    MatIconModule,
    RouterModule,
    MatProgressBarModule,
    TranslateModule,
    MatTabsModule,
    TotalNonConformanceListDetailsComponent,
    TotalCapaListDetailsComponent,
  ],
  templateUrl: './file-upload-question.component.html',
  styleUrls: ['./file-upload-question.component.scss'],
})
export class FileUploadQuestionComponent
  extends BaseComponent
  implements OnInit, OnDestroy {
  @Input() question!: AuditTemplateQuestion;
  @Input() isActive: boolean = false;
  @Input() auditId: string | undefined = '';
  @Input() auditStatus?: AUDIT_STATUS;
  @Output() responseChanged = new EventEmitter<any>();
  @Output() fetchReviewers = new EventEmitter<void>();
  @Output() openCAPADialog = new EventEmitter<{ auditId: string, auditResponseId: string }>();
  @Output() openNCDialog = new EventEmitter<{ auditId: string, auditResponseId: string }>();
  allowFileExtension: AllowFileExtension[] = [];
  files: string[] = [];
  errorMessage: string | null = null;
  isDragging = false;
  documentId: string | null = null;
  resultArray: {
    isSuccess: boolean;
    documentId: string;
    message: string;
    name: string;
    totalChunk?: number;
  }[] = [];
  uploadProgress = 0;
  auditResponse: AuditResponse = {} as AuditResponse;
  fileUploadGroup!: FormGroup;
  fb = inject(FormBuilder);
  toastrService = inject(ToastrService);
  auditStore = inject(AuditStore);
  cd = inject(ChangeDetectorRef);
  extension: string | null = null;
  loading = false;
  counter = 0;
  chunkSize = environment.chunkSize;
  chunkUploads: Observable<any>[] = [];
  fileArray: any[] = [];
  AUDIT_STATUS = AUDIT_STATUS;
  overlay = inject(OverlayPanel);

  constructor(
    private auditService: AuditService,
    private commonService: CommonService,
    private commonDialogService: CommonDialogService
  ) {
    super();
  }

  ngOnInit(): void {
    this.createDocumentForm();
    this.getAllAllowFileExtension();
    this.loadExistingResponse();

    this.auditService.updateTrigger.subscribe((value) => {
      if (value) {
        this.loadExistingResponse();
      }
    });
  }

  private loadExistingResponse(): void {
    this.sub$.sink = this.auditService
      .getResponseForQuestion(this.auditId ?? '', this.question.id ?? '')
      .subscribe((response) => {
        if (response) {
          this.auditResponse = response;
          this.fileUploadGroup.get('score')?.setValue(this.auditResponse.score || 0);
          this.fileArray = response?.auditResponseAttachments?.map((file) => ({
            id: file.id,
            isSuccess: true,
            name: file.fileName,
            extension: file.extension,
            totalChunk: file.totalChunk,
          })) ?? [];
          this.responseChanged.emit(this.auditResponse);
        }
      });
  }

  get allowedFileTypes(): string {
    const allowedExtensionscombine = this.allowFileExtension.map(
      (ext) => ext.extension
    );
    return allowedExtensionscombine.length > 0
      ? allowedExtensionscombine.join(',')
      : '';
  }

  get fileInputs(): FormArray {
    return (<FormArray>this.fileUploadGroup.get('files')) as FormArray;
  }

  createDocumentForm() {
    this.fileUploadGroup = this.fb.group({
      files: this.fb.array([]),
      score: ['', [Validators.required, Validators.pattern('^[0-9]+$'), Validators.max(this.question.maxScore || 0)]],
    });
  }

  getAllAllowFileExtension() {
    this.sub$.sink = this.commonService.allowFileExtension$.subscribe(
      (allowFileExtension: AllowFileExtension[]) => {
        if (allowFileExtension) {
          this.allowFileExtension = allowFileExtension;
        }
      }
    );
  }

  onDragOver(event: DragEvent): void {
    event.preventDefault();
    event.stopPropagation();
    this.isDragging = true;
  }

  onDragLeave(): void {
    this.isDragging = false;
  }

  async onDrop(event: DragEvent) {
    event.preventDefault();
    event.stopPropagation();
    this.isDragging = false;

    if (event.dataTransfer?.files) {
      await this.fileUpload(event.dataTransfer?.files);
      // this.handleFiles(event.dataTransfer.files);
    }
  }

  async onFileSelected(event: Event) {
    const input = event.target as HTMLInputElement;
    await this.fileUpload(input.files ?? undefined);
    // Reset the input to allow selecting the same file again
    input.value = '';
  }
  fileExtesionValidation(extension: string): boolean {
    const allowTypeExtenstion = this.allowFileExtension.find((c) =>
      c.extensions?.find((ext) => ext.toLowerCase() === extension.toLowerCase())
    );
    return allowTypeExtenstion ? true : false;
  }

  async onFileSelect(event: Event) {
    const input = event.target as HTMLInputElement;
    if (input.files) {
      await this.fileUpload(input.files);
      this.cd.markForCheck();
    }
  }

  async fileUpload(files: FileList | undefined) {
    if (!files) return;

    for (let i = 0; i < files.length; i++) {
      if (!(await validateFile(files[i]))) {
        this.toastrService.error(
          this.translationService.getValue(
            'INVALID_EXTENSION_OR_CORRUPT_INVALID_SIGNATURE'
          )
        );
        this.cd.markForCheck();
        continue;
      }
      const file = files[i];
      this.extension = file.name.split('.').pop() ?? '';
      if (this.fileExtesionValidation(this.extension)) {
        this.fileInputs.push(
          this.fb.group({
            fileName: [file.name],
            file: [file],
            name: [file.name, Validators.required],
            extension: [this.extension],
            message: [''],
            isSuccess: [false],
            isLoading: [false],
            auditResponseId: [''],
            auditResponseAttachmentId: [''],
            totalChunk: [Math.ceil(file.size / this.chunkSize)],
          })
        );
      }
    }
  }

  removeLocalFile(index: number): void {
    if (index >= 0 && index < this.fileInputs.length) {
      this.fileInputs.removeAt(index);
    }
  }
  saveAuditResponse() {
    if (this.fileUploadGroup.invalid) {
      this.fileUploadGroup.markAllAsTouched();
      return;
    }
    if (this.fileInputs.length === 0) {
      this.toastrService.error(
        this.translationService.getValue('PLEASE_ADD_FILE')
      );
      return;
    }

    // Collect all file names: existing (already uploaded) + new (to be uploaded)
    const existingFileNames = this.fileArray.map(file => file.name);
    const newFileNames = this.fileInputs.controls.map(control => control.get('fileName')?.value);
    // Merge and deduplicate file names
    const allFileNames = Array.from(new Set([...existingFileNames, ...newFileNames]));

    // Remove file names that have been deleted (i.e., not present in fileArray or fileInputs)
    const auditQuestionResponse: AuditQuestionResponse = {
      id: this.auditResponse ? this.auditResponse.id : '',
      auditId: this.auditId ?? '',
      questionId: this.question.id ?? '',
      response: allFileNames.join(','),
      score: this.fileUploadGroup.get('score')?.value || 0,
    };
    if (auditQuestionResponse.id) {
      this.auditService
        .updateAuditQuestionResponse(auditQuestionResponse)
        .subscribe({
          next: () => {
            this.toastrService.success(
              this.translationService.getValue('RESPONSE_UPDATED_SUCCESSFULLY')
            );
            this.saveFilesAndDocument(auditQuestionResponse.id ?? '');
            this.responseChanged.emit(auditQuestionResponse);
          },
          error: () => {
            this.toastrService.error(
              this.translationService.getValue('FAILED_TO_UPDATE_RESPONSE')
            );
          },
        });
    } else {
      this.auditService
        .saveAuditQuestionResponse(auditQuestionResponse)
        .subscribe({
          next: (auditQuestionResponse: AuditQuestionResponse) => {
            this.toastrService.success(
              this.translationService.getValue('RESPONSE_SAVED_SUCCESSFULLY')
            );
            this.auditResponse = auditQuestionResponse;
            this.saveFilesAndDocument(auditQuestionResponse.id ?? '');
            const audit: StartAudit = {
              id: this.auditId || '',
              auditTemplateId: this.question?.auditTemplateId || '',
              status: AUDIT_STATUS.INPROGRESS,
            };
            this.auditStore.updateAudit(audit);
          },
          error: () => {
            this.toastrService.error(
              this.translationService.getValue('FAILED_TO_SAVE_RESPONSE')
            );
          },
        });
    }
  }

  saveAuditResponseAttachmentChunk(
    auditRespAttachment: AuditResponseAttachment
  ): Observable<AuditResponseAttachment | CommonError> {
    return this.auditService.saveAuditResponseAttachment(auditRespAttachment);
  }

  saveFilesAndDocument(auditResponseId: string): void {
    if (!this.fileInputs.valid) {
      this.fileInputs.markAllAsTouched();
      return;
    }

    const fileInputsArray = this.fileInputs.controls;
    if (fileInputsArray.length === 0) {
      this.loading = false;
      this.toastrService.error(
        this.translationService.getValue('PLEASE_ADD_FILE')
      );
      return;
    }

    const concatObservable$: Observable<any>[] = [];
    fileInputsArray.forEach((control) => {
      if (!control.get('isSuccess')?.value) {
        const auditResponseAttachment: AuditResponseAttachment = {
          fileName: control?.get('fileName')?.value ?? '',
          extension: control?.get('extension')?.value ?? '',
          totalChunk: control?.get('totalChunk')?.value ?? 0,
          auditId: this.auditId ?? '',
          auditResponseId: auditResponseId ?? '',
          fileSize: this.formatFileSize(control?.get('file')?.value.size),
        };
        concatObservable$.push(
          this.saveAuditResponseAttachmentChunk({ ...auditResponseAttachment })
        );
      }
    });

    if (concatObservable$.length === 0) {
      this.loading = false;
      return;
    }

    this.resultArray = [];
    this.chunkUploads = [];
    this.counter = 0;

    this.sub$.sink = from(concatObservable$)
      .pipe(
        concatMap((obs) =>
          obs.pipe(
            catchError((err) => of(`${err.error?.[0] || 'Upload failed'}`))
          )
        )
      )
      .subscribe({
        next: (document: AuditResponseAttachment) => {
          this.counter++;
          if (typeof document !== 'string') {
            this.fileInputs.at(this.counter - 1).patchValue({
              isLoading: false,
              auditResponseAttachmentId: document.id,
              isSuccess: true,
            });
            const fileData = this.fileInputs?.at(this.counter - 1)?.get('file')?.value;
            const totalChunk = this.fileInputs?.at(this.counter - 1)?.get('totalChunk')?.value;
            this.chunkUploads.push(this.uploadFile(document.id ?? '', fileData, totalChunk));
          } else {
            this.resultArray.push({
              isSuccess: false,
              documentId: '',
              message: document,
              name: this.fileInputs?.at(this.counter - 1)?.get('name')?.value,
              totalChunk: this.fileInputs?.at(this.counter - 1)?.get('totalChunk')?.value,
            });
            this.fileInputs.at(this.counter - 1).patchValue({
              isLoading: false,
              isSuccess: false,
            });
          }
        },
        error: () => {
          if (this.counter < this.fileInputs.length) {
            this.fileInputs.at(this.counter).patchValue({
              isLoading: false,
              auditResponseAttachmentId: '',
            });
          }
          this.loading = false;
        },
        complete: () => {
          if (this.chunkUploads.length > 0) {
            this.uploadedChunksAllFile();
            this.loading = false;
          }
        },
      });
  }

  uploadFile(auditResponseAttachmentId: string, file: File, totalChunk: number): Observable<any> {
    const { parallelCalls } = this.commonService.getNetworkSpeed();
    const chunkUploads: Observable<any>[] = [];
    this.uploadProgress = 0;
    for (let i = 0; i < totalChunk; i++) {
      const start = i * this.chunkSize;
      const end = Math.min(start + this.chunkSize, file.size);
      const chunk = file.slice(start, end);
      const formData = new FormData();
      formData.append('file', chunk);
      formData.append('chunkIndex', i.toString());
      formData.append('size', this.chunkSize.toString());
      formData.append('totalChunks', totalChunk.toString());
      formData.append('extension', this.extension ?? '');
      formData.append('auditResponseAttachmentId', auditResponseAttachmentId);
      chunkUploads.push(this.uploadChunk(formData));
    }
    return from(chunkUploads).pipe(
      mergeMap((upload) => upload, parallelCalls), // Uploads max 5 chunks at a time
      toArray(),
      concatMap(() => {
        return this.completeUpload(auditResponseAttachmentId);
      }),
      catchError((err) => {
        const documentChunkStatus: DocumentChunkStatus = {
          documentId: auditResponseAttachmentId,
          status: false,
        };
        return of(documentChunkStatus);
      })
    );
  }

  completeUpload(auditResponseAttachmentId: string): Observable<any> {
    return this.auditService.markAsAuditResponseAttachmentChunkComplete(
      auditResponseAttachmentId,
      true
    );
  }

  uploadChunk(
    formData: FormData
  ): Observable<AuditResponseAttachmentChunk | CommonError> {
    return this.auditService.uploadAuditResponseAttachmentChunk(formData);
  }

  removeFile(file: any, index: number): void {
    this.sub$.sink = this.commonDialogService
      .deleteConfirmtionDialog(
        `${this.translationService.getValue(
          'ARE_YOU_SURE_YOU_WANT_TO_DELETE'
        )} ${file.name}`
      )
      .subscribe((isTrue: boolean) => {
        if (isTrue) {
          this.sub$.sink = this.auditService
            .deleteAuditResponseAttachment(file.id)
            .subscribe(() => {
              this.fileArray.splice(index, 1);
              this.toastrService.success(
                this.translationService.getValue(
                  'AUDIT_RESPONSE_ATTACHMENT_DELETED_SUCCESSFULLY'
                )
              );
              this.cd.detectChanges();
            });
        }
      });
  }

  getFileIcon(fileType: string): string {
    if (fileType.startsWith('image/')) {
      return 'image';
    } else if (fileType.includes('pdf')) {
      return 'picture_as_pdf';
    } else if (fileType.includes('spreadsheet') || fileType.includes('excel')) {
      return 'table_chart';
    } else if (fileType.includes('document') || fileType.includes('word')) {
      return 'description';
    } else {
      return 'insert_drive_file';
    }
  }

  formatFileSize(bytes: number): number {
    return parseInt((bytes / 1024).toFixed(1));
  }

  documentChunkRemove(documentId: string) {
    this.loading = true;
    this.sub$.sink = this.auditService
      .markAsAuditResponseAttachmentChunkComplete(documentId, false)
      .subscribe({
        next: (data) => {
          this.loading = false;
        },
        error: (error) => {
          this.loading = false;
        },
      });
  }

  uploadedChunksAllFile(): void {
    this.sub$.sink = from(this.chunkUploads)
      .pipe(
        concatMap((obs) =>
          obs.pipe(
            catchError((err) => of(`${err.error[0] || 'Chunk upload failed'}`))
          )
        )
      )
      .subscribe({
        next: (data: AuditResponseChunkStatus) => {
          this.loading = false;
          if (data.status) {
            const fileIndex = this.fileInputs.controls.findIndex(
              (ctrl) =>
                ctrl.get('auditResponseAttachmentId')?.value ===
                data.auditResponseAttachmentId
            );
            if (fileIndex !== -1) {
              const control = this.fileInputs.at(fileIndex);
              this.resultArray.push({
                isSuccess: true,
                documentId: data.auditResponseAttachmentId,
                name: control.get('name')?.value,
                message: this.translationService.getValue(
                  'DOCUMENT_SAVE_SUCCESSFULLY'
                ),
              });
            }
          } else {
            this.documentChunkRemove(data.auditResponseAttachmentId);
          }
        },
        error: () => {
          this.loading = false;
        },
        complete: () => {
          this.uploadProgress = 100;
          this.fileInputs.clear();
          const newFiles = this.resultArray
            .filter((f) => f.isSuccess)
            .map((file) => ({
              id: file.documentId,
              isSuccess: true,
              name: file.name,
              extension: file.name.split('.').pop(),
              totalChunk: file.totalChunk,
            }));
          this.fileArray = [...this.fileArray, ...newFiles];
          this.cd.detectChanges();
          this.fetchReviewers.emit();
          this.loadExistingResponse();
        },
      });
  }

  onMediaView(document: AuditResponseAttachment) {
    const documentView: DocumentView = {
      documentId: document.id,
      name: document.name,
      extension: document.extension,
      isVersion: false,
      isFromPublicPreview: false,
      isPreviewDownloadEnabled: true,
      isFileRequestDocument: false,
      totalChunk: document.totalChunk,
      moduleNo: QmsModuleEnum.Audit,
    };
    this.overlay.open(BasePreviewComponent, {
      position: 'center',
      origin: 'global',
      panelClass: ['file-preview-overlay-container', 'white-background'],
      data: documentView,
    });
  }

  openCapaRequestDialog(auditId: string, auditResponseId: string): void {
    this.openCAPADialog.emit({ auditId, auditResponseId });
  }

  openNonConformanceDialog(auditId: string, auditResponseId: string): void {
    this.openNCDialog.emit({ auditId, auditResponseId });
  }
}
