import { Component, EventEmitter, inject, Input, OnInit, Output, ViewChild } from '@angular/core';
import {
  FormBuilder,
  FormControl,
  FormGroup,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { MatCheckboxModule } from '@angular/material/checkbox';
import { MatRadioModule } from '@angular/material/radio';
import { MatFormFieldModule } from '@angular/material/form-field';
import { AuditService } from '../../audit.service';
import { QUESTION_TYPE } from '../../../audit-template/models/question-type';
import { AuditTemplateQuestion } from '../../../audit-template/models/audit-template-question';
import { AuditResponse } from '../../models/audit-response';
import { BaseComponent } from '../../../base.component';
import { MatIconModule } from '@angular/material/icon';
import { MatButtonModule } from '@angular/material/button';
import { AuditQuestionResponse } from '@core/domain-classes/audit-question-respone';

import { ToastrService } from '@core/services/toastr-service';
import { StartAudit } from '@core/domain-classes/audit';
import { AUDIT_STATUS } from '../../../audit-template/models/audit-status';
import { AuditStore } from '../../audit-store';
import { RouterModule } from '@angular/router';
import { TranslateModule } from '@ngx-translate/core';
import { TotalCapaListDetailsComponent } from '../../audit-response-details/total-capa-list-details/total-capa-list-details.component';
import { TotalNonConformanceListDetailsComponent } from '../../audit-response-details/total-non-conformance-list-details/total-non-conformance-list-details.component';
import { MatTabsModule } from '@angular/material/tabs';
import { EvidenceAttachmentsComponent } from '../evidence-attachments/evidence-attachments.component';

@Component({
  selector: 'app-multi-select-question',
  standalone: true,
  imports: [
    ReactiveFormsModule,
    MatCheckboxModule,
    MatRadioModule,
    RouterModule,
    TranslateModule,
    MatButtonModule,
    MatFormFieldModule,
    MatTabsModule,
    TotalNonConformanceListDetailsComponent,
    TotalCapaListDetailsComponent,
    MatIconModule,
    EvidenceAttachmentsComponent
  ],
  templateUrl: './multi-select-question.component.html',
  styleUrls: ['./multi-select-question.component.scss'],
})
export class MultiSelectQuestionComponent
  extends BaseComponent
  implements OnInit {
  @ViewChild(EvidenceAttachmentsComponent, { static: true })
  evidenceAttachmentsComponent!: EvidenceAttachmentsComponent;
  @Input() question!: AuditTemplateQuestion;
  @Input() isActive: boolean = false;
  @Input() auditStatus?: AUDIT_STATUS;
  @Input() auditId: string | undefined = '';
  @Output() responseChanged = new EventEmitter<any>();
  @Output() fetchReviewers = new EventEmitter<void>();
  @Output() openCAPADialog = new EventEmitter<{ auditId: string, auditResponseId: string }>();
  @Output() openNCDialog = new EventEmitter<{ auditId: string, auditResponseId: string }>();

  QUESTION_TYPE = QUESTION_TYPE;
  auditResponse: AuditResponse = {} as AuditResponse;
  form!: FormGroup;
  errorMessage: string | null = null;
  auditStore = inject(AuditStore);
  AUDIT_STATUS = AUDIT_STATUS;

  constructor(
    private toastrService: ToastrService,
    private fb: FormBuilder,
    private auditService: AuditService
  ) {
    super();
  }

  ngOnInit(): void {
    this.initForm();
    this.loadExistingResponse();

    this.auditService.updateTrigger.subscribe((value) => {
      if (value) {
        this.loadExistingResponse();
      }
    });
  }

  private initForm(): void {
    if (this.question.inputType === QUESTION_TYPE.MULTI_SELECT) {
      // Create a form with a checkbox for each option
      const controls: Record<string, FormControl> = {};

      this.question?.questionOptions?.forEach((_, index) => {
        controls[`option_${index}`] = this.fb.control(false);
      });

      controls['score'] = this.fb.control(0, [Validators.required, Validators.pattern('^[0-9]+$'), Validators.max(this.question.maxScore || 0),]);

      this.form = this.fb.group(controls);
    } else {
      // Create a form with a radio button group
      this.form = this.fb.group({
        selectedOption: [''],
        requiredEvidence: [this.question.requiredEvidence],
        score: ['', [
          Validators.required,
          Validators.pattern('^[0-9]+$'),
          Validators.max(this.question.maxScore || 0),
        ]],
      });
    }
  }

  private loadExistingResponse(): void {
    this.sub$.sink = this.auditService
      .getResponseForQuestion(this.auditId ?? '', this.question.id ?? '')
      .subscribe((response: AuditResponse) => {
        if (response) {
          this.auditResponse = response;
          this.form.get('score')?.setValue(response.score);
          this.responseChanged.emit(this.auditResponse);
          if (this.question.inputType == QUESTION_TYPE.MULTI_SELECT) {
            // Set checkbox values
            const selectedValues = response.response.split(',') || [];
            this.question?.questionOptions?.forEach((option, index) => {
              const control = this.form.get(`option_${index}`);
              if (control) {
                control.setValue(selectedValues.includes(option.id ?? ''));
              }
            });
          } else {
            // Set radio value
            this.form.get('selectedOption')?.setValue(response.response);
          }
        }
      });
  }

  onCheckboxChange(option: string, isChecked: boolean): void {
    // Get currently selected values
    const selectedValues = this.getSelectedValues();

    // Update the selection
    if (isChecked && !selectedValues.includes(option)) {
      selectedValues.push(option);
    } else if (!isChecked && selectedValues.includes(option)) {
      const index = selectedValues.indexOf(option);
      selectedValues.splice(index, 1);
    }
  }

  private getSelectedValues(): string[] {
    const selectedValues: string[] = [];

    if (this.question.inputType == QUESTION_TYPE.MULTI_SELECT) {
      this.question?.questionOptions?.forEach((option, index) => {
        if (this.form.get(`option_${index}`)?.value) {
          selectedValues.push(option.id ?? '');
        }
      });
    } else {
      const selectedOption = this.form.get('selectedOption')?.value;
      if (selectedOption) {
        selectedValues.push(selectedOption);
      }
    }

    return selectedValues;
  }

  saveResponse(): void {
    // Mark all controls as touched to trigger validation
    if (!this.form.valid) {
      this.form.markAllAsTouched();
      return;
    }
    if (!this.evidenceAttachmentsComponent.checkFilesValidation()) {
      this.toastrService.error(
        this.translationService.getValue('EVIDENCE_IS_REQUIRED')
      );
      return;
    }
    const selectedValues: string[] = this.getSelectedValues();
    // Validate the response
    if (this.form.invalid || selectedValues.length === 0) {
      this.errorMessage = 'This question requires an answer';
      return;
    }
    this.errorMessage = null;
    const auditQuestionResponse: AuditQuestionResponse = {
      id: this.auditResponse ? this.auditResponse.id : '',
      auditId: this.auditId ?? '',
      questionId: this.question.id ?? '',
      response: selectedValues.join(','),
      score: this.form.get('score')?.value || 0,
    };
    if (auditQuestionResponse.id) {
      this.auditService
        .updateAuditQuestionResponse(auditQuestionResponse)
        .subscribe({
          next: () => {
            this.evidenceAttachmentsComponent.saveFilesAndDocument(auditQuestionResponse.id ?? '');
            this.toastrService.success(
              this.translationService.getValue('RESPONSE_UPDATED_SUCCESSFULLY')
            );
          },
          error: () => {
            this.toastrService.error(
              this.translationService.getValue('FAILED_TO_UPDATE_RESPONSE')
            );
          },
        });
    } else {
      this.auditService
        .saveAuditQuestionResponse(auditQuestionResponse)
        .subscribe({
          next: (auditQuestionResponse: AuditQuestionResponse) => {
            this.evidenceAttachmentsComponent.saveFilesAndDocument(auditQuestionResponse.id ?? '');
            this.toastrService.success(
              this.translationService.getValue('RESPONSE_SAVED_SUCCESSFULLY')
            );
            // this.fetchReviewers.emit();
            // const audit: StartAudit = {
            //   id: this.auditId || '',
            //   auditTemplateId: this.question?.auditTemplateId || '',
            //   status: AUDIT_STATUS.INPROGRESS,
            // };
            // this.auditStore.updateAudit(audit);
            // this.loadExistingResponse();
          },
          error: () => {
            this.toastrService.error(
              this.translationService.getValue('FAILED_TO_SAVE_RESPONSE')
            );
          },
        });
    }
  }
  onEvidenceAttachments(questionId?: string): void {
    if (questionId == this.question.id) {
      this.fetchReviewers.emit();
      if (this.auditStatus !== AUDIT_STATUS.INPROGRESS) {
        const audit: StartAudit = {
          id: this.auditId || '',
          auditTemplateId: this.question?.auditTemplateId || '',
          status: AUDIT_STATUS.INPROGRESS,
        };
        this.auditStore.updateAudit(audit);
      }
      this.loadExistingResponse();
    }
  }

  openCapaRequestDialog(auditId: string, auditResponseId: string): void {
    this.openCAPADialog.emit({ auditId, auditResponseId });
  }

  openNonConformanceDialog(auditId: string, auditResponseId: string): void {
    this.openNCDialog.emit({ auditId, auditResponseId });
  }
}
