import {
  Component,
  EventEmitter,
  inject,
  Input,
  OnInit,
  Output,
  ViewChild,
} from '@angular/core';
import {
  FormBuilder,
  FormGroup,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { MatIconModule } from '@angular/material/icon';
import { MatInputModule } from '@angular/material/input';
import { TranslateModule } from '@ngx-translate/core';
import { AuditTemplateQuestion } from '../../../audit-template/models/audit-template-question';
import { BaseComponent } from '../../../base.component';
import { AuditService } from '../../audit.service';
import { AuditResponse } from '../../models/audit-response';
import { MatButtonModule } from '@angular/material/button';
import { AuditQuestionResponse } from '@core/domain-classes/audit-question-respone';

import { ToastrService } from '@core/services/toastr-service';
import { StartAudit } from '@core/domain-classes/audit';
import { AUDIT_STATUS } from '../../../audit-template/models/audit-status';
import { AuditStore } from '../../audit-store';
import { RouterModule } from '@angular/router';
import { MatTabsModule } from '@angular/material/tabs';
import { TotalCapaListDetailsComponent } from '../../audit-response-details/total-capa-list-details/total-capa-list-details.component';
import { TotalNonConformanceListDetailsComponent } from '../../audit-response-details/total-non-conformance-list-details/total-non-conformance-list-details.component';
import { EvidenceAttachmentsComponent } from '../evidence-attachments/evidence-attachments.component';

@Component({
  selector: 'app-number-question',
  standalone: true,
  imports: [
    ReactiveFormsModule,
    TranslateModule,
    MatInputModule,
    RouterModule,
    MatIconModule,
    MatButtonModule,
    MatTabsModule,
    TotalNonConformanceListDetailsComponent,
    TotalCapaListDetailsComponent,
    EvidenceAttachmentsComponent
  ],
  templateUrl: './number-question.component.html',
  styleUrl: './number-question.component.scss',
})
export class NumberQuestionComponent extends BaseComponent implements OnInit {
  @ViewChild(EvidenceAttachmentsComponent, { static: true })
  evidenceAttachmentsComponent!: EvidenceAttachmentsComponent;
  @Input() question!: AuditTemplateQuestion;
  @Input() isActive: boolean = false;
  @Input() auditStatus?: AUDIT_STATUS;
  @Input() auditId: string | undefined = '';
  @Output() responseChanged = new EventEmitter<any>();
  @Output() fetchReviewers = new EventEmitter<void>();
  @Output() openCAPADialog = new EventEmitter<{ auditId: string, auditResponseId: string }>();
  @Output() openNCDialog = new EventEmitter<{ auditId: string, auditResponseId: string }>();
  auditService = inject(AuditService);
  auditStore = inject(AuditStore);
  auditResponse: AuditResponse = {} as AuditResponse;
  form!: FormGroup;
  errorMessage: string | null = null;
  AUDIT_STATUS = AUDIT_STATUS;
  constructor(
    private toastrService: ToastrService,
    private fb: FormBuilder
  ) {
    super();
  }

  ngOnInit(): void {
    this.initForm();
    this.loadExistingResponse();

    this.auditService.updateTrigger.subscribe((value) => {
      if (value) {
        this.loadExistingResponse();
      }
    });
  }

  private initForm(): void {
    this.form = this.fb.group({
      response: [
        null,
        [Validators.required, Validators.pattern('^-?\\d*\\.?\\d+$')],
      ],
      requiredEvidence: [this.question.requiredEvidence],
      score: ['', [Validators.required, Validators.pattern('^[0-9]+$'), Validators.max(this.question.maxScore || 0)]],
    });
  }

  private loadExistingResponse(): void {
    this.sub$.sink = this.auditService
      .getResponseForQuestion(this.auditId ?? '', this.question.id ?? '')
      .subscribe((response: AuditResponse) => {
        if (response) {
          this.auditResponse = response;
          this.form.get('score')?.setValue(this.auditResponse.score);
          this.form.get('response')?.setValue(this.auditResponse.response);
          this.responseChanged.emit(this.auditResponse);
        }
      });
  }

  saveResponse(): void {
    if (!this.form.valid) {
      this.form.markAllAsTouched();
      return;
    }
    if (!this.evidenceAttachmentsComponent.checkFilesValidation()) {
      this.toastrService.error(
        this.translationService.getValue('PLEASE_RECORD_RESPONSE')
      );
      return;
    }
    const value = this.form.get('response')?.value;
    const auditQuestionResponse: AuditQuestionResponse = {
      id: this.auditResponse ? this.auditResponse.id : '',
      auditId: this.auditId ?? '',
      questionId: this.question.id ?? '',
      response: value.toString(),
      score: this.form.get('score')?.value || 0,
    };
    if (auditQuestionResponse.id) {
      this.auditService
        .updateAuditQuestionResponse(auditQuestionResponse)
        .subscribe({
          next: () => {
            this.evidenceAttachmentsComponent.saveFilesAndDocument(auditQuestionResponse.id ?? '');
            this.toastrService.success(
              this.translationService.getValue('RESPONSE_UPDATED_SUCCESSFULLY')
            );
          },
          error: () => {
            this.toastrService.error(
              this.translationService.getValue('FAILED_TO_UPDATE_RESPONSE')
            );
          },
        });
    } else {
      this.auditService
        .saveAuditQuestionResponse(auditQuestionResponse)
        .subscribe({
          next: (auditQuestionResponse: AuditQuestionResponse) => {
            this.evidenceAttachmentsComponent.saveFilesAndDocument(auditQuestionResponse.id ?? '');
            this.toastrService.success(
              this.translationService.getValue('RESPONSE_SAVED_SUCCESSFULLY')
            );
          },
          error: () => {
            this.toastrService.error(
              this.translationService.getValue('FAILED_TO_SAVE_RESPONSE')
            );
          },
        });
    }
  }
  onEvidenceAttachments(questionId?: string): void {
    if (questionId == this.question.id) {
      this.fetchReviewers.emit();
      if (this.auditStatus !== AUDIT_STATUS.INPROGRESS) {
        const audit: StartAudit = {
          id: this.auditId || '',
          auditTemplateId: this.question?.auditTemplateId || '',
          status: AUDIT_STATUS.INPROGRESS,
        };
        this.auditStore.updateAudit(audit);
      }
      this.loadExistingResponse();
    }
  }

  openCapaRequestDialog(auditId: string, auditResponseId: string): void {
    this.openCAPADialog.emit({ auditId, auditResponseId });
  }

  openNonConformanceDialog(auditId: string, auditResponseId: string): void {
    this.openNCDialog.emit({ auditId, auditResponseId });
  }
}
