import {
  Component,
  EventEmitter,
  inject,
  Input,
  OnInit,
  Output,
  ViewChild,
} from '@angular/core';
import {
  FormBuilder,
  FormGroup,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { MatSliderModule } from '@angular/material/slider';
import { MatIconModule } from '@angular/material/icon';
import { AuditService } from '../../audit.service';
import { AuditTemplateQuestion } from '../../../audit-template/models/audit-template-question';
import { BaseComponent } from '../../../base.component';
import { AuditResponse } from '../../models/audit-response';
import { MatButtonModule } from '@angular/material/button';
import { AuditQuestionResponse } from '@core/domain-classes/audit-question-respone';
import { TranslateModule } from '@ngx-translate/core';

import { ToastrService } from '@core/services/toastr-service';
import { StartAudit } from '@core/domain-classes/audit';
import { AUDIT_STATUS } from '../../../audit-template/models/audit-status';
import { AuditStore } from '../../audit-store';
import { RouterModule } from '@angular/router';
import { MatTabsModule } from '@angular/material/tabs';
import { TotalCapaListDetailsComponent } from '../../audit-response-details/total-capa-list-details/total-capa-list-details.component';
import { TotalNonConformanceListDetailsComponent } from '../../audit-response-details/total-non-conformance-list-details/total-non-conformance-list-details.component';
import { EvidenceAttachmentsComponent } from '../evidence-attachments/evidence-attachments.component';

@Component({
  selector: 'app-rating-question',
  standalone: true,
  imports: [
    ReactiveFormsModule,
    MatSliderModule,
    MatIconModule,
    RouterModule,
    TranslateModule,
    MatButtonModule,
    MatTabsModule,
    TotalNonConformanceListDetailsComponent,
    TotalCapaListDetailsComponent,
    EvidenceAttachmentsComponent
  ],
  templateUrl: './rating-question.component.html',
  styleUrls: ['./rating-question.component.scss'],
})
export class RatingQuestionComponent extends BaseComponent implements OnInit {
  @ViewChild(EvidenceAttachmentsComponent, { static: true })
  evidenceAttachmentsComponent!: EvidenceAttachmentsComponent;
  @Input() question!: AuditTemplateQuestion;
  @Input() isActive: boolean = false;
  @Input() auditId: string | undefined = '';
  @Input() auditStatus?: AUDIT_STATUS;
  @Output() responseChanged = new EventEmitter<any>();
  @Output() fetchReviewers = new EventEmitter<void>();
  auditService = inject(AuditService);
  auditStore = inject(AuditStore);
  auditResponse: AuditResponse = {} as AuditResponse;
  form!: FormGroup;
  errorMessage: string | null = null;
  currentRating = 0;
  hoverRating = 0;
  stars: number[] = [];
  minRating: number = 1;
  maxRating: number = 5;
  AUDIT_STATUS = AUDIT_STATUS;
  @Output() openCAPADialog = new EventEmitter<{ auditId: string, auditResponseId: string }>();
  @Output() openNCDialog = new EventEmitter<{ auditId: string, auditResponseId: string }>();

  constructor(
    private fb: FormBuilder,
    private toastrService: ToastrService,
  ) {
    super();
  }

  ngOnInit(): void {
    const option = this.question?.questionOptions && this.question.questionOptions.length > 0
      ? this.question.questionOptions[0]
      : undefined;

    const optionValue = option?.optionValue ? parseInt(option.optionValue) : undefined;
    const optionText = option?.optionText ? parseInt(option.optionText) : undefined;

    this.stars = Array(optionValue || 5)
      .fill(0)
      .map((_, i) => i);
    this.minRating = optionText || 1;
    this.maxRating = optionValue || 5;
    this.initForm();
    this.loadExistingResponse();

    this.auditService.updateTrigger.subscribe((value) => {
      if (value) {
        this.loadExistingResponse();
      }
    });
  }

  private initForm(): void {
    this.form = this.fb.group({
      rating: [
        null,
        [
          Validators.required,
          Validators.min(this.minRating),
          Validators.max(this.maxRating),
        ],
      ],
      requiredEvidence: [this.question.requiredEvidence],
      score: ['', [
        Validators.required,
        Validators.pattern('^[0-9]+$'),
        Validators.max(this.question.maxScore || 0),
      ]],
    });
  }

  private loadExistingResponse(): void {
    this.sub$.sink = this.auditService
      .getResponseForQuestion(this.auditId ?? '', this.question.id ?? '')
      .subscribe((auditResponse: AuditResponse) => {
        if (auditResponse) {
          this.auditResponse = auditResponse;
          this.form.get('score')?.setValue(this.auditResponse.score);
          this.responseChanged.emit(this.auditResponse);
          this.currentRating = parseInt(auditResponse.response);
          this.form.get('rating')?.setValue(this.currentRating);
        }
      });
  }

  setRating(rating: number): void {
    this.currentRating = rating;
    this.form.get('rating')?.setValue(rating);
  }

  onSliderChange(value: number): void {
    this.currentRating = value;
  }

  saveResponse(): void {
    if (!this.form.valid) {
      this.form.markAllAsTouched();
      this.errorMessage = this.translationService.getValue(
        'PLEASE_SELECT_A_RATING'
      );
      // Clear error message when form value changes
      this.form.valueChanges.pipe().subscribe(() => {
        this.errorMessage = null;
      });
      return;
    } else {
      this.errorMessage = null;
    }
    if (!this.evidenceAttachmentsComponent.checkFilesValidation()) {
      this.toastrService.error(
        this.translationService.getValue('PLEASE_SELECT_RATING')
      );
      return;
    }
    const value = this.form.get('rating')?.value;
    const auditQuestionResponse: AuditQuestionResponse = {
      id: this.auditResponse ? this.auditResponse.id : '',
      auditId: this.auditId ?? '',
      questionId: this.question.id ?? '',
      response: value.toString(),
      score: this.form.get('score')?.value || 0,
    };
    if (auditQuestionResponse.id) {
      this.auditService
        .updateAuditQuestionResponse(auditQuestionResponse)
        .subscribe({
          next: () => {
            this.evidenceAttachmentsComponent.saveFilesAndDocument(auditQuestionResponse.id ?? '');
            this.toastrService.success(
              this.translationService.getValue('RESPONSE_UPDATED_SUCCESSFULLY')
            );
          },
          error: () => {
            this.toastrService.error(
              this.translationService.getValue('FAILED_TO_UPDATE_RESPONSE')
            );
          },
        });
    } else {
      this.auditService
        .saveAuditQuestionResponse(auditQuestionResponse)
        .subscribe({
          next: (auditQuestionResponse: AuditQuestionResponse) => {
            this.evidenceAttachmentsComponent.saveFilesAndDocument(auditQuestionResponse.id ?? '');
            this.toastrService.success(
              this.translationService.getValue('RESPONSE_SAVED_SUCCESSFULLY')
            );
          },
          error: () => {
            this.toastrService.error(
              this.translationService.getValue('FAILED_TO_SAVE_RESPONSE')
            );
          },
        });
    }
  }

  onEvidenceAttachments(questionId?: string): void {
    if (questionId == this.question.id) {
      this.fetchReviewers.emit();
      if (this.auditStatus !== AUDIT_STATUS.INPROGRESS) {
        const audit: StartAudit = {
          id: this.auditId || '',
          auditTemplateId: this.question?.auditTemplateId || '',
          status: AUDIT_STATUS.INPROGRESS,
        };
        this.auditStore.updateAudit(audit);
      }
      this.loadExistingResponse();
    }
  }

  openCapaRequestDialog(auditId: string, auditResponseId: string): void {
    this.openCAPADialog.emit({ auditId, auditResponseId });
  }

  openNonConformanceDialog(auditId: string, auditResponseId: string): void {
    this.openNCDialog.emit({ auditId, auditResponseId });
  }
}
