import { Component, EventEmitter, inject, Input, OnInit, Output, ViewChild } from '@angular/core';
import { FormBuilder, FormGroup, ReactiveFormsModule, Validators } from '@angular/forms';
import { MatRadioModule } from '@angular/material/radio';
import { MatSlideToggleModule } from '@angular/material/slide-toggle';
import { MatButtonToggleModule } from '@angular/material/button-toggle';
import { AuditService } from '../../audit.service';
import { AuditTemplateQuestion } from '../../../audit-template/models/audit-template-question';
import { BaseComponent } from '../../../base.component';
import { AuditResponse } from '../../models/audit-response';
import { MatIconModule } from '@angular/material/icon';
import { MatButtonModule } from '@angular/material/button';
import { AuditQuestionResponse } from '@core/domain-classes/audit-question-respone';

import { ToastrService } from '@core/services/toastr-service';
import { StartAudit } from '@core/domain-classes/audit';
import { AUDIT_STATUS } from '../../../audit-template/models/audit-status';
import { AuditStore } from '../../audit-store';
import { RouterModule } from '@angular/router';
import { TranslateModule } from '@ngx-translate/core';
import { TotalCapaListDetailsComponent } from '../../audit-response-details/total-capa-list-details/total-capa-list-details.component';
import { MatCardModule } from '@angular/material/card';
import { MatTabsModule } from '@angular/material/tabs';
import { TotalNonConformanceListDetailsComponent } from '../../audit-response-details/total-non-conformance-list-details/total-non-conformance-list-details.component';
import { EvidenceAttachmentsComponent } from '../evidence-attachments/evidence-attachments.component';

@Component({
  selector: 'app-yes-no-question',
  standalone: true,
  imports: [
    ReactiveFormsModule,
    MatRadioModule,
    MatSlideToggleModule,
    RouterModule,
    MatButtonToggleModule,
    TranslateModule,
    MatIconModule,
    MatButtonModule,
    MatCardModule,
    MatTabsModule,
    TotalNonConformanceListDetailsComponent,
    TotalCapaListDetailsComponent,
    EvidenceAttachmentsComponent
  ],
  templateUrl: './yes-no-question.component.html',
  styleUrls: ['./yes-no-question.component.scss'],
})
export class YesNoQuestionComponent extends BaseComponent implements OnInit {
  @ViewChild(EvidenceAttachmentsComponent, { static: true })
  evidenceAttachmentsComponent!: EvidenceAttachmentsComponent;
  @Input() question!: AuditTemplateQuestion;
  @Input() auditId: string | undefined = '';
  @Input() isActive: boolean = false;
  @Input() auditStatus?: AUDIT_STATUS;
  AUDIT_STATUS = AUDIT_STATUS;
  @Output() responseChanged = new EventEmitter<any>();
  @Output() fetchReviewers = new EventEmitter<void>();
  form!: FormGroup;
  auditResponse: AuditResponse = {} as AuditResponse;
  auditStore = inject(AuditStore);
  @Output() openCAPADialog = new EventEmitter<{ auditId: string, auditResponseId: string }>();
  @Output() openNCDialog = new EventEmitter<{ auditId: string, auditResponseId: string }>();

  constructor(
    private fb: FormBuilder,
    private toastrService: ToastrService,
    private auditService: AuditService
  ) {
    super();
  }

  ngOnInit(): void {
    this.initForm();
    this.loadExistingResponse();

    this.auditService.updateTrigger.subscribe((value) => {
      if (value) {
        this.loadExistingResponse();
      }
    });
  }

  private initForm(): void {
    this.form = this.fb.group({
      answer: [null, [Validators.required]],
      requiredEvidence: [this.question.requiredEvidence],
      score: ['', [Validators.required, Validators.pattern('^[0-9]+$'), Validators.max(this.question.maxScore || 0)]],
    });
  }

  private loadExistingResponse(): void {
    this.sub$.sink = this.auditService
      .getResponseForQuestion(this.auditId ?? '', this.question.id ?? '')
      .subscribe((auditResponse: AuditResponse) => {
        if (auditResponse) {
          this.auditResponse = auditResponse;
          this.form.get('score')?.setValue(auditResponse.score);
          this.form.get('answer')?.setValue(auditResponse.response);
          this.responseChanged.emit(auditResponse);
        }
      });
  }

  saveResponse(): void {
    if (this.form.invalid) {
      this.form.markAllAsTouched();
      return;
    }
    if (!this.evidenceAttachmentsComponent.checkFilesValidation()) {
      this.toastrService.error(
        this.translationService.getValue('PLEASE_SELECT_YES_OR_NO')
      );
      return;
    }
    const auditQuestionResponse: AuditQuestionResponse = {
      id: this.auditResponse ? this.auditResponse.id : '',
      auditId: this.auditId ?? '',
      questionId: this.question.id ?? '',
      response: this.form.get('answer')?.value,
      score: this.form.get('score')?.value || 0,
    };
    if (auditQuestionResponse.id) {
      this.auditService
        .updateAuditQuestionResponse(auditQuestionResponse)
        .subscribe({
          next: () => {
            this.evidenceAttachmentsComponent.saveFilesAndDocument(auditQuestionResponse.id ?? '');
            this.toastrService.success(
              this.translationService.getValue('RESPONSE_UPDATED_SUCCESSFULLY')
            );
          },
          error: () => {
            this.toastrService.error(
              this.translationService.getValue('FAILED_TO_UPDATE_RESPONSE')
            );
          },
        });
    } else {
      this.auditService
        .saveAuditQuestionResponse(auditQuestionResponse)
        .subscribe({
          next: (auditQuestionResponse: AuditQuestionResponse) => {
            this.evidenceAttachmentsComponent.saveFilesAndDocument(auditQuestionResponse.id ?? '');
            this.toastrService.success(
              this.translationService.getValue('RESPONSE_SAVED_SUCCESSFULLY')
            );
          },
          error: () => {
            this.toastrService.error(
              this.translationService.getValue('FAILED_TO_SAVE_RESPONSE')
            );
          },
        });
    }
  }
  onEvidenceAttachments(questionId?: string): void {
    if (questionId == this.question.id) {
      this.fetchReviewers.emit();
      if (this.auditStatus !== AUDIT_STATUS.INPROGRESS) {
        const audit: StartAudit = {
          id: this.auditId || '',
          auditTemplateId: this.question?.auditTemplateId || '',
          status: AUDIT_STATUS.INPROGRESS,
        };
        this.auditStore.updateAudit(audit);
      }
      this.loadExistingResponse();
    }
  }

  openCapaRequestDialog(auditId: string, auditResponseId: string): void {
    this.openCAPADialog.emit({ auditId, auditResponseId });
  }

  openNonConformanceDialog(auditId: string, auditResponseId: string): void {
    this.openNCDialog.emit({ auditId, auditResponseId });
  }
}
