import { Component, inject, Inject, OnInit } from '@angular/core';
import { FormBuilder, FormGroup, ReactiveFormsModule, Validators } from '@angular/forms';
import { MatButtonModule } from '@angular/material/button';
import { MAT_DIALOG_DATA, MatDialogModule, MatDialogRef } from '@angular/material/dialog';
import { MatSelectModule } from '@angular/material/select';
import { Router } from '@angular/router';
import { SubmitAudit } from '@core/domain-classes/audit';

import { ToastrService } from '@core/services/toastr-service';
import { AUDIT_STATUS } from '../../audit-template/models/audit-status';
import { StartAuditComponent } from '../../audit-template/start-audit/start-audit.component';
import { BaseComponent } from '../../base.component';
import { UserStore } from '../../user/store/user.store';
import { AuditReviewerService } from './audit-reviewer.service';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { TranslateModule } from '@ngx-translate/core';
import { AuditStore } from '../audit-store';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';

@Component({
  selector: 'app-submit-audit',
  imports: [
    MatDialogModule,
    ReactiveFormsModule,
    MatButtonModule,
    MatSelectModule,
    PageHelpTextComponent,
    TranslateModule,
    MatIconModule,
    MatCardModule
  ],
  templateUrl: './submit-audit.component.html',
  styleUrl: './submit-audit.component.scss'
})
export class SubmitAuditComponent extends BaseComponent implements OnInit {
  auditForm: FormGroup;
  userStore = inject(UserStore);
  auditStore = inject(AuditStore);
  constructor(
    private fb: FormBuilder,
    private toastrService: ToastrService,
    private auditReviewerService: AuditReviewerService,
    private dialogRef: MatDialogRef<StartAuditComponent>,
    @Inject(MAT_DIALOG_DATA) public data: { auditId: string },
    private router: Router
  ) {
    super();
    this.data = { auditId: data.auditId };
  }
  ngOnInit(): void {
    this.createForm();
  }

  createForm(): void {
    this.auditForm = this.fb.group({
      reviewerId: ['', [Validators.required]],
      auditId: [this.data.auditId],
      comments: [''],
    });
  }

  saveAudit(): void {
    if (this.auditForm.invalid) {
      this.auditForm.markAllAsTouched();
      return;
    }
    const reviewerId = this.auditForm.get('reviewerId')?.value;
    const audit: SubmitAudit = {
      reviewerId: reviewerId,
      auditId: this.data.auditId,
      status: AUDIT_STATUS.INREVIEW,
      comments: this.auditForm.get('comments')?.value,
    };
    this.auditReviewerService.submitAudit(audit).subscribe({
      next: (response) => {
        this.toastrService.success(
          this.translationService.getValue('SUBMIT_AUDIT_SUCCESSFULLY')
        );
        this.auditStore.setloadListFlag(true);
        this.dialogRef.close();
        this.router.navigate(['/audit', 'report', this.data.auditId]);
      }
    });
  }

  closeDialog() {
    this.dialogRef.close();
  }
}
