import { Component, inject, Input, OnInit, SimpleChanges, OnChanges } from '@angular/core';
import { CapaAction } from '../../model/capa-action';
import { MatTableModule } from '@angular/material/table';
import { BaseComponent } from '../../../base.component';

import { ToastrService } from '@core/services/toastr-service';
import { CapaActionService } from '../capa-action.service';
import { MatButtonModule } from '@angular/material/button';
import { MatDialog } from '@angular/material/dialog';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { ManageCapaActionComponent } from '../manage-capa-action/manage-capa-action.component';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { TranslateModule } from '@ngx-translate/core';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { ActivatedRoute } from '@angular/router';
import { CapaActionTypePipe } from '../../pipes/capa-action-type.pipe';
import { CapaActionStatusPipe } from '../../pipes/capa-action-status.pipe';
import { NgClass } from '@angular/common';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';

@Component({
  selector: 'app-capa-action-list',
  imports: [
    MatTableModule,
    MatButtonModule,
    HasClaimDirective,
    TranslateModule,
    UTCToLocalTime,
    CapaActionTypePipe,
    CapaActionStatusPipe,
    NgClass,
    MatIconModule,
    MatCardModule
  ],
  templateUrl: './capa-action-list.component.html',
  styleUrl: './capa-action-list.component.scss'
})
export class CapaActionListComponent extends BaseComponent implements OnInit, OnChanges {
  @Input() capaActionResponse: CapaAction | null = null;

  capaActionList: CapaAction[] = [];
  displayedColumns: string[] = ['action', 'actionType', 'assignTo', 'actionStatus', 'dueDate'];
  toastrService = inject(ToastrService);
  capaActionService = inject(CapaActionService);
  dialog = inject(MatDialog);
  commandDialogService = inject(CommonDialogService);
  route = inject(ActivatedRoute);

  ngOnInit(): void {
    this.route.parent?.paramMap.subscribe(params => {
      const capaRequestId = params.get('id');
      if (capaRequestId) {
        this.getAllActions(capaRequestId);
      }
    });
  }

  ngOnChanges(changes: SimpleChanges): void {
    if (changes['capaActionResponse'] && this.capaActionResponse) {
      if (this.capaActionResponse.id) {
        this.capaActionList = [...this.capaActionList, this.capaActionResponse];
      }
    }
  }

  getAllActions(capaRequestId: string) {
    this.sub$.sink = this.capaActionService.getCapaActionsByRequestId(capaRequestId).subscribe({
      next: (actions) => {
        const capaActions = actions as CapaAction[];
        if (capaActions) {
          this.capaActionList = capaActions;
        }
      },
      error: (error) => {
        this.toastrService.error(error);
      }
    });
  }

  viewCapaAction(capaAction: CapaAction) {
    const dialogRef = this.dialog.open(ManageCapaActionComponent, {
      maxWidth: '60vw',
      width: '100%',
      data: { capaAction, isViewOnly: true },
    });

    dialogRef.afterClosed().subscribe((result: CapaAction | null) => {
      if (result) {
        const existingIndex = this.capaActionList.findIndex(action => action.id === result.id);
        if (existingIndex !== -1) {
          this.capaActionList = [
            ...this.capaActionList.slice(0, existingIndex),
            result,
            ...this.capaActionList.slice(existingIndex + 1)
          ];
        } else {
          this.capaActionList = [...this.capaActionList, result];
        }
      }
    });
  }

  deleteCapaAction(capaActionId: string) {
    this.commandDialogService.deleteConfirmtionDialog(this.translationService.getValue("ARE_YOU_SURE_YOU_WANT_TO_DELETE_THIS_CAPA_ACTION")).subscribe({
      next: (result: boolean) => {
        if (result) {
          this.sub$.sink = this.capaActionService.deleteCapaAction(capaActionId).subscribe({
            next: () => {
              this.toastrService.success(this.translationService.getValue('CAPA_ACTION_DELETE_SUCCESSFULLY'));
              this.capaActionList = this.capaActionList.filter(action => action.id !== capaActionId);
            },
            error: (error) => {
              this.toastrService.error(error);
            }
          });
        }
      }
    });
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.capaActionList.indexOf(row);
  }
}
