import { ChangeDetectorRef, Component, Inject, inject, OnInit, Optional } from '@angular/core';
import { CapaActionListComponent } from "../capa-action-list/capa-action-list.component";
import { ActivatedRoute, Router, RouterLink } from '@angular/router';
import { FormBuilder, FormGroup, ReactiveFormsModule, Validators } from '@angular/forms';
import { TextEditorComponent } from "../../../shared/text-editor/text-editor.component";
import { MatSelectModule } from '@angular/material/select';
import { MatCardModule } from '@angular/material/card';
import { CAPA_ACTION_STATUS } from '../../model/capa-action-status-enum';
import { MatButtonModule } from '@angular/material/button';
import { BaseComponent } from '../../../base.component';
import { MatRadioModule } from '@angular/material/radio';
import { CAPA_ACTION_TYPE } from '../../model/capa-action-type-enum';
import { CommonService } from '@core/services/common.service';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { CapaActionService } from '../capa-action.service';

import { ToastrService } from '@core/services/toastr-service';
import { CapaAction } from '../../model/capa-action';
import { UserStore } from '../../../user/store/user.store';
import { MAT_DIALOG_DATA, MatDialogRef } from '@angular/material/dialog';
import { NgClass } from '@angular/common';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { TranslateModule, TranslateService } from '@ngx-translate/core';
import { MatIconModule } from '@angular/material/icon';
import { RichTextRequired } from '@shared/validators/rich-text.validator';
import { TranslationService } from '@core/services/translation.service';
import { CdkAutofill } from "@angular/cdk/text-field";

@Component({
  selector: 'app-manage-capa-action',
  imports: [
    CapaActionListComponent,
    MatIconModule,
    TextEditorComponent,
    ReactiveFormsModule,
    MatSelectModule,
    MatCardModule,
    MatButtonModule,
    MatRadioModule,
    MatDatepickerModule,
    NgClass,
    HasClaimDirective,
    TranslateModule
  ],
  templateUrl: './manage-capa-action.component.html',
  styleUrl: './manage-capa-action.component.scss'
})
export class ManageCapaActionComponent extends BaseComponent implements OnInit {
  capaActionForm: FormGroup;
  fb = inject(FormBuilder);
  userStore = inject(UserStore);
  capaActionResponse: CapaAction = {} as CapaAction;
  commonService = inject(CommonService);
  route = inject(ActivatedRoute);
  capaActionService = inject(CapaActionService);
  toastrService = inject(ToastrService);
  translateService = inject(TranslationService);
  router = inject(Router);
  minDueDate: Date = new Date();

  statusList = Object.keys(CAPA_ACTION_STATUS)
    .filter(key => !isNaN(Number(CAPA_ACTION_STATUS[key as any])))
    .map(key => ({
      label: key,
      value: CAPA_ACTION_STATUS[key as keyof typeof CAPA_ACTION_STATUS]
    }));
  actionType = Object.keys(CAPA_ACTION_TYPE)
    .filter(key => !isNaN(Number(CAPA_ACTION_TYPE[key as any])))
    .map(key => ({
      label: key,
      value: CAPA_ACTION_TYPE[key as keyof typeof CAPA_ACTION_TYPE]
    }));

  constructor(
    @Optional() private dialogRef: MatDialogRef<ManageCapaActionComponent>,
    @Optional() @Inject(MAT_DIALOG_DATA) public data: { capaAction: CapaAction, isViewOnly: boolean },
  ) {
    super();
  }

  ngOnInit(): void {
    this.createCapaActionForm();
    this.capaActionForm.get('actionType')?.setValue(0);

    this.route.parent?.paramMap.subscribe(params => {
      const capaRequestId = params.get('id');
      if (capaRequestId) {
        this.capaActionForm.get('capaRequestId')?.setValue(capaRequestId);
      }
    });

    if (this.data !== null && this.data.capaAction) {
      this.capaActionForm.patchValue(this.data.capaAction);
    } else {
      this.data = {
        capaAction: {} as CapaAction,
        isViewOnly: false
      };
    }
  }

  createCapaActionForm(): void {
    this.capaActionForm = this.fb.group({
      id: [''],
      capaRequestId: [''],
      actionType: [],
      assignToId: [''],
      status: ['', [Validators.required]],
      dueDate: [''],
      actionDescription: ['', [RichTextRequired]],
    });
  }

  onSubmit(): void {
    if (this.capaActionForm.invalid) {
      this.capaActionForm.markAllAsTouched();
      return;
    };

    const capaAction = this.capaActionForm.getRawValue();

    if (capaAction.capaRequestId && this.data.capaAction.id) {
      this.capaActionService.updateCapaAction(capaAction).subscribe({
        next: (response) => {
          this.toastrService.success(this.translateService.getValue('CAPA_ACTION_UPDATE_SUCCESSFULLY'));
          this.dialogRef.close(response);
        }
      });
    } else if (capaAction.capaRequestId) {
      this.capaActionService.addCapaAction(capaAction).subscribe({
        next: (response: CapaAction) => {
          this.capaActionResponse = response;
          this.toastrService.success(this.translateService.getValue('CAPA_ACTION_CREATE_SUCCESSFULLY'));
          this.capaActionForm.reset();
          this.capaActionForm.get('actionType')?.setValue(0);
          this.capaActionForm.get('capaRequestId')?.setValue(this.capaActionResponse.capaRequestId);
        }
      });
    }
  }

  onCancel(): void {
    if (this.data.isViewOnly) {
      this.dialogRef.close();
    } else {
      const capaRequestId = this.capaActionForm.get('capaRequestId')?.value;
      if (capaRequestId) {
        this.router.navigate(['/capa/details', capaRequestId], { queryParams: { isDetailsMode: true } });
      }
    }
  }
}
