import { HttpClient } from '@angular/common/http';
import { inject, Injectable } from '@angular/core';
import { CapaRequestAttachment } from '../model/capa-request-attachment';
import { catchError, Observable, retry, timer } from 'rxjs';
import { CommonError } from '@core/error-handler/common-error';
import { CommonHttpErrorService } from '@core/error-handler/common-http-error.service';
import { CapaRequestAttachmentChunk } from '../model/capa-request-attachment-chunk';

@Injectable({
  providedIn: 'root'
})
export class CapaAttachmentService {
  maxRetries: number = 2; // Maximum number of retries for chunk uploads
  private httpClient = inject(HttpClient);
  private commonHttpErrorService = inject(CommonHttpErrorService);

  saveCapaAttachment(response: CapaRequestAttachment): Observable<CapaRequestAttachment | CommonError> {
    const url = 'CapaAttachment';
    return this.httpClient
      .post<CapaRequestAttachment>(url, response)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  uploadCapaAttachmentChunk(capaAttachmentChunk: FormData): Observable<CapaRequestAttachmentChunk | CommonError> {
    const url = 'CapaAttachment/Chunk';
    return this.httpClient
      .post<CapaRequestAttachmentChunk>(url, capaAttachmentChunk)
      .pipe(
        retry({
          count: this.maxRetries, // Retry up to 2 times
          delay: (error, retryCount) => {
            console.warn(`Retrying chunk... Attempt ${retryCount}`);
            return timer(Math.pow(2, retryCount) * 1000);
          },
        })
      );
  }

  markAsCapaAttachmentChunkComplete(capaAttachmentChunkId: string, status: boolean): Observable<void | CommonError> {
    const url = 'CapaAttachment/Chunk/uploadStatus';
    return this.httpClient
      .post<void>(url, { capaAttachmentId: capaAttachmentChunkId, status: status, })
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getAllCapaAttachmentsByRequestId(requestId: string): Observable<CapaRequestAttachment[]> {
    const url = `CapaAttachment/${requestId}`;
    return this.httpClient
      .get<CapaRequestAttachment[]>(url);

  }

  deleteUploadedFile(capaAttachmentId: string): Observable<void | CommonError> {
    const url = `CapaAttachment/${capaAttachmentId}`;
    return this.httpClient
      .delete<void>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }
}
