import { Component, inject, OnInit, ViewChild } from '@angular/core';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { MatPaginator } from '@angular/material/paginator';
import { MatSort, MatSortModule } from '@angular/material/sort';
import { MatTableModule } from '@angular/material/table';
import { ActivatedRoute, RouterModule } from '@angular/router';
import { BaseComponent } from '../../base.component';
import { CapaRequest } from '../model/capa-request';
import { IdName } from '@core/domain-classes/id-name';
import { CommonError } from '@core/error-handler/common-error';
import { PageSizeOption } from '@core/utils/global-config';
import { Subject, debounceTime, distinctUntilChanged, merge, tap } from 'rxjs';
import { UserStore } from '../../user/store/user.store';
import { CapaRequestService } from '../capa-request/capa-request.service';
import { CAPA_REQUEST_PRIORITY } from '../model/capa-Request-Priority-enum';
import { CAPA_REQUEST_STATUS } from '../model/capa-Request-Status-enum';
import { FormsModule } from '@angular/forms';
import { MatSelectModule } from '@angular/material/select';
import { MatIconModule } from '@angular/material/icon';
import { CapaRequestResource } from '../model/capa-request-resource';
import { HttpResponse } from '@angular/common/http';
import { CapaRequestLogService } from './capa-request-log.service';
import { CAPA_REQUEST_LOG_STATUS } from '../model/capa-request-log-status.enum';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { TranslateModule } from '@ngx-translate/core';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { CapaRequestStatusPipe } from '../pipes/capa-request-status.pipe';
import { CapaRequestPriorityPipe } from '../pipes/capa-request-priority.pipe';
import { NgClass } from '@angular/common';
import { CapaRequestLogStatusPipe } from '../pipes/capa-request-log-status.pipe';
import { MatCardModule } from '@angular/material/card';

@Component({
  selector: 'app-capa-request-log-list',
  imports: [
    MatTableModule,
    MatDatepickerModule,
    MatPaginator,
    FormsModule,
    MatSelectModule,
    MatIconModule,
    MatSortModule,
    RouterModule,
    PageHelpTextComponent,
    TranslateModule,
    UTCToLocalTime,
    CapaRequestStatusPipe,
    CapaRequestPriorityPipe,
    NgClass,
    CapaRequestLogStatusPipe,
    MatCardModule
  ],
  templateUrl: './capa-request-log-list.component.html',
  styleUrl: './capa-request-log-list.component.scss'
})
export class CapaRequestLogListComponent extends BaseComponent implements OnInit {
  capaRequestList: CapaRequest[] = [];
  displayedColumns: string[] = ['referenceNo', 'title', 'raiseById', 'priority', 'status', 'dueDate', 'audit', 'ncNumber', 'logStatus'];
  displayedColumnSecondary: string[] = ['search-referenceNo', 'search-title', 'search-raiseById', 'search-priority', 'search-status', 'search-dueDate', 'search-audit', 'search-ncNumber', 'search-logStatus'];
  footerToDisplayed: string[] = ['footer'];
  userStore = inject(UserStore);
  auditList: IdName[] = [];
  minDate: Date = new Date();
  capaRquestLogService = inject(CapaRequestLogService);
  route = inject(ActivatedRoute);
  pageOption = PageSizeOption;
  capaRequestLogService = inject(CapaRequestLogService);
  capaRquestService = inject(CapaRequestService);
  capaRequestFilterParameter: CapaRequestResource = {
    orderBy: 'referenceNo asc',
    pageSize: 10,
    skip: 0,
    totalCount: 0,
    referenceNo: '',
    title: '',
    priority: '',
    raisedById: '',
    auditId: '',
    capaRequestStatus: '',
    dueStartDate: undefined,
    dueEndDate: undefined,
    logStatus: ''
  };
  filterParameter$: Subject<string> = new Subject<string>();
  _raisedByIdFilter = '';
  _priorityFilter = '';
  _statusFilter = '';
  _referenceNoFilter = '';
  _auditIdFilter = '';
  _titleFilter = '';
  _fromDueDateFilter: Date | null = null;
  _toDueDateFilter: Date | null = null;
  _logStatusFilter = '';

  @ViewChild(MatPaginator) paginator: MatPaginator;
  @ViewChild(MatSort) sort: MatSort;

  statusList = Object.keys(CAPA_REQUEST_STATUS)
    .filter((key) => !isNaN(Number(CAPA_REQUEST_STATUS[key as any])))
    .map((key) => ({
      name: key,
      id: CAPA_REQUEST_STATUS[key as keyof typeof CAPA_REQUEST_STATUS],
    }));
  priorityList = Object.keys(CAPA_REQUEST_PRIORITY)
    .filter((key) => !isNaN(Number(CAPA_REQUEST_PRIORITY[key as any])))
    .map((key) => ({
      name: key,
      id: CAPA_REQUEST_PRIORITY[key as keyof typeof CAPA_REQUEST_PRIORITY],
    }));
  logStatus = Object.keys(CAPA_REQUEST_LOG_STATUS)
    .filter((key) => !isNaN(Number(CAPA_REQUEST_LOG_STATUS[key as any])))
    .map((key) => ({
      id: key,
      name: CAPA_REQUEST_LOG_STATUS[key as keyof typeof CAPA_REQUEST_LOG_STATUS],
    }));

  public get referenceNoFilter(): string {
    return this._referenceNoFilter;
  }

  public set referenceNoFilter(v: string) {
    if (this._referenceNoFilter !== v) {
      this._referenceNoFilter = v;
      const referenceNoFilter = `referenceNo#${v}`;
      this.filterParameter$.next(referenceNoFilter);
    }
  }

  public get titleFilter(): string {
    return this._titleFilter;
  }

  public set titleFilter(v: string) {
    if (this._titleFilter !== v) {
      this._titleFilter = v;
      const titleFilter = `title#${v}`;
      this.filterParameter$.next(titleFilter);
    }
  }

  public get raisedByIdFilter(): string {
    return this._raisedByIdFilter;
  }

  public set raisedByIdFilter(v: string) {
    if (this._raisedByIdFilter !== v) {
      this._raisedByIdFilter = v;
      const raisedByIdFilter = `raisedById#${v}`;
      this.filterParameter$.next(raisedByIdFilter);
    }
  }

  public get auditIdFilter(): string {
    return this._auditIdFilter;
  }

  public set auditIdFilter(v: string) {
    if (this._auditIdFilter !== v) {
      this._auditIdFilter = v;
      const auditIdFilter = `auditId#${v}`;
      this.filterParameter$.next(auditIdFilter);
    }
  }

  public get statusFilter(): string {
    return this._statusFilter;
  }

  public set statusFilter(v: string) {
    if (this._statusFilter !== v) {
      this._statusFilter = v;
      const statusFilter = `status#${v}`;
      this.filterParameter$.next(statusFilter);
    }
  }

  public get priorityFilter(): string {
    return this._priorityFilter;
  }

  public set priorityFilter(v: string) {
    if (this._priorityFilter !== v) {
      this._priorityFilter = v;
      const priorityFilter = `priority#${v}`;
      this.filterParameter$.next(priorityFilter);
    }
  }

  public get dueFromDateFilter(): Date | undefined {
    return this._fromDueDateFilter ?? undefined;
  }

  public set dueFromDateFilter(v: Date) {
    if (this._fromDueDateFilter !== v) {
      this._fromDueDateFilter = v;
      const fromDateFilter = `fromDate#${v}`;
      this.filterParameter$.next(fromDateFilter);
    }
  }

  public get dueToDateFilter(): Date | undefined {
    return this._toDueDateFilter ?? undefined;
  }

  public set dueToDateFilter(v: Date) {
    if (this._toDueDateFilter !== v) {
      this._toDueDateFilter = v;
      const toDateFilter = `toDate#${v}`;
      this.filterParameter$.next(toDateFilter);
    }
  }

  public get logStatusFilter(): string {
    return this._logStatusFilter;
  }

  public set logStatusFilter(v: string) {
    if (this._logStatusFilter !== v) {
      this._logStatusFilter = v;
      const logStatusFilter = `logStatus#${v}`;
      this.filterParameter$.next(logStatusFilter);
    }
  }

  ngOnInit(): void {
    this.getAllAudits();
    this.loadByQuery(this.capaRequestFilterParameter);

    this.sub$.sink = this.filterParameter$
      .pipe(debounceTime(1000), distinctUntilChanged())
      .subscribe((c: string) => {
        this.capaRequestFilterParameter.skip = 0;
        this.paginator.pageIndex = 0;
        const filterArray: Array<string> = c.split('#');
        if (filterArray[0] === 'referenceNo') {
          this.capaRequestFilterParameter.referenceNo = filterArray[1];
        } else if (filterArray[0] === 'title') {
          this.capaRequestFilterParameter.title = filterArray[1];
        } else if (filterArray[0] === 'raisedById') {
          this.capaRequestFilterParameter.raisedById = filterArray[1];
        } else if (filterArray[0] === 'auditId') {
          this.capaRequestFilterParameter.auditId = filterArray[1];
        } else if (filterArray[0] === 'priority') {
          this.capaRequestFilterParameter.priority = filterArray[1];
        } else if (filterArray[0] === 'status') {
          this.capaRequestFilterParameter.capaRequestStatus = filterArray[1];
        } else if (filterArray[0] === 'logStatus') {
          this.capaRequestFilterParameter.logStatus = filterArray[1];
        } else if (filterArray[0] === 'fromDate') {
          if (filterArray[1] != 'null') {
            this.capaRequestFilterParameter.dueStartDate = new Date(
              filterArray[1]
            );
            this.capaRequestFilterParameter.dueEndDate = this.dueToDateFilter;
          } else {
            this.capaRequestFilterParameter.dueStartDate = undefined;
            this.capaRequestFilterParameter.dueEndDate = undefined;
          }
        } else if (filterArray[0] === 'toDate') {
          if (filterArray[1] != 'null') {
            this.capaRequestFilterParameter.dueEndDate = new Date(
              filterArray[1]
            );
            this.capaRequestFilterParameter.dueStartDate =
              this.dueFromDateFilter;
          } else {
            this.capaRequestFilterParameter.dueStartDate = undefined;
            this.capaRequestFilterParameter.dueEndDate = undefined;
          }
        }
        this.loadByQuery(this.capaRequestFilterParameter);
      });
  }

  clearOrderDates() {
    this._fromDueDateFilter = null;
    this._toDueDateFilter = null;
  }

  getAllAudits() {
    this.capaRquestService.getAllAudits().subscribe({
      next: (result: IdName[] | CommonError) => {
        if (Array.isArray(result)) {
          this.auditList = result;
        } else {
          this.auditList = [];
        }
      },
    });
  }

  ngAfterViewInit(): void {
    this.sub$.sink = this.sort.sortChange.subscribe(() => {
      this.paginator.pageIndex = 0;
    });
    this.sub$.sink = merge(this.sort.sortChange, this.paginator.page)
      .pipe(
        tap(() => {
          this.capaRequestFilterParameter.skip =
            this.paginator.pageIndex * this.paginator.pageSize;
          this.capaRequestFilterParameter.pageSize = this.paginator.pageSize;
          this.capaRequestFilterParameter.orderBy =
            this.sort.active + ' ' + this.sort.direction;
          this.loadByQuery(this.capaRequestFilterParameter);
        })
      ).subscribe();
  }

  loadByQuery(filter: CapaRequestResource) {
    return this.capaRequestLogService.getCapaRequests(filter).subscribe({
      next: (httpResponse: HttpResponse<CapaRequest[]>) => {
        if (httpResponse && httpResponse.body) {
          this.capaRequestList = [...httpResponse.body as CapaRequest[]];
        }
        const paginationHeader = httpResponse.headers.get('x-pagination');
        if (paginationHeader) {
          const pagination = JSON.parse(paginationHeader);
          this.capaRequestFilterParameter = { ...filter, totalCount: pagination.totalCount };
        }
      },
    })
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.capaRequestList.indexOf(row);
  }
}
