import {
  AfterViewInit,
  Component,
  inject,
  OnInit,
  ViewChild,
} from '@angular/core';
import { ActivatedRoute, RouterLink } from '@angular/router';
import { CapaRequest } from '../../model/capa-request';
import { MatTableModule } from '@angular/material/table';
import { CapaRequestStore } from '../capa-request.store';
import { PageSizeOption } from '@core/utils/global-config';
import { MatButtonModule } from '@angular/material/button';
import { MatMenuModule } from '@angular/material/menu';
import { MatIconModule } from '@angular/material/icon';
import { MatPaginator } from '@angular/material/paginator';
import { MatSort, MatSortModule } from '@angular/material/sort';
import { MatSelectModule } from '@angular/material/select';
import { BaseComponent } from '../../../base.component';
import { CAPA_REQUEST_STATUS } from '../../model/capa-Request-Status-enum';
import { CAPA_REQUEST_PRIORITY } from '../../model/capa-Request-Priority-enum';
import { CommonService } from '@core/services/common.service';
import { debounceTime, distinctUntilChanged, merge, Subject, tap } from 'rxjs';
import { FormsModule } from '@angular/forms';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { UserStore } from '../../../user/store/user.store';
import { IdName } from '@core/domain-classes/id-name';
import { CapaRequestService } from '../capa-request.service';
import { CommonError } from '@core/error-handler/common-error';
import { TranslateModule } from '@ngx-translate/core';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { CapaRequestPriorityPipe } from '../../pipes/capa-request-priority.pipe';
import { CapaRequestStatusPipe } from '../../pipes/capa-request-status.pipe';
import { MatCardModule } from '@angular/material/card';
import { NgClass } from '@angular/common';

@Component({
  selector: 'app-capa-request-list',
  imports: [
    RouterLink,
    MatTableModule,
    MatButtonModule,
    MatMenuModule,
    MatIconModule,
    MatPaginator,
    MatSelectModule,
    MatDatepickerModule,
    FormsModule,
    MatSortModule,
    TranslateModule,
    PageHelpTextComponent,
    HasClaimDirective,
    UTCToLocalTime,
    CapaRequestPriorityPipe,
    CapaRequestStatusPipe,
    MatCardModule,
    NgClass
  ],
  templateUrl: './capa-request-list.component.html',
  styleUrl: './capa-request-list.component.scss',
})
export class CapaRequestListComponent
  extends BaseComponent
  implements OnInit, AfterViewInit {
  capaRequestList: CapaRequest[] = [];
  displayedColumns: string[] = [
    'action',
    'referenceNo',
    'title',
    'raiseById',
    'priority',
    'status',
    'dueDate',
    'ncNumber',
    'audit',
  ];
  displayedColumnSecondary: string[] = [
    'search-action',
    'search-referenceNo',
    'search-title',
    'search-raiseById',
    'search-priority',
    'search-status',
    'search-dueDate',
    'search-ncNumber',
    'search-audit',
  ];
  footerToDisplayed: string[] = ['footer'];
  capaRequestStore = inject(CapaRequestStore);
  userStore = inject(UserStore);
  auditList: IdName[] = [];
  capaRquestService = inject(CapaRequestService);
  route = inject(ActivatedRoute);
  pageOption = PageSizeOption;
  commonService = inject(CommonService);
  commandDialogService = inject(CommonDialogService);
  capaRequestFilterParameter = { ...this.capaRequestStore.filterParameters() };
  filterParameter$: Subject<string> = new Subject<string>();
  _raisedByIdFilter = this.capaRequestFilterParameter.raisedById;
  _priorityFilter = this.capaRequestFilterParameter.priority;
  _statusFilter = this.capaRequestFilterParameter.capaRequestStatus;
  _referenceNoFilter = this.capaRequestFilterParameter.referenceNo;
  _auditIdFilter = this.capaRequestFilterParameter.auditId;
  _titleFilter = this.capaRequestFilterParameter.title;
  _fromDueDateFilter = this.capaRequestFilterParameter.dueStartDate;
  _toDueDateFilter = this.capaRequestFilterParameter.dueEndDate;

  @ViewChild(MatPaginator) paginator: MatPaginator;
  @ViewChild(MatSort) sort: MatSort;

  statusList = Object.keys(CAPA_REQUEST_STATUS)
    .filter((key) => !isNaN(Number(CAPA_REQUEST_STATUS[key as any])))
    .map((key) => ({
      name: key,
      id: CAPA_REQUEST_STATUS[key as keyof typeof CAPA_REQUEST_STATUS],
    }));
  priorityList = Object.keys(CAPA_REQUEST_PRIORITY)
    .filter((key) => !isNaN(Number(CAPA_REQUEST_PRIORITY[key as any])))
    .map((key) => ({
      name: key,
      id: CAPA_REQUEST_PRIORITY[key as keyof typeof CAPA_REQUEST_PRIORITY],
    }));

  public get referenceNoFilter(): string {
    return this._referenceNoFilter;
  }

  public set referenceNoFilter(v: string) {
    if (this._referenceNoFilter !== v) {
      this._referenceNoFilter = v;
      const referenceNoFilter = `referenceNo#${v}`;
      this.filterParameter$.next(referenceNoFilter);
    }
  }

  public get titleFilter(): string {
    return this._titleFilter;
  }

  public set titleFilter(v: string) {
    if (this._titleFilter !== v) {
      this._titleFilter = v;
      const titleFilter = `title#${v}`;
      this.filterParameter$.next(titleFilter);
    }
  }

  public get raisedByIdFilter(): string {
    return this._raisedByIdFilter;
  }

  public set raisedByIdFilter(v: string) {
    if (this._raisedByIdFilter !== v) {
      this._raisedByIdFilter = v;
      const raisedByIdFilter = `raisedById#${v}`;
      this.filterParameter$.next(raisedByIdFilter);
    }
  }

  public get auditIdFilter(): string {
    return this._auditIdFilter;
  }

  public set auditIdFilter(v: string) {
    if (this._auditIdFilter !== v) {
      this._auditIdFilter = v;
      const auditIdFilter = `auditId#${v}`;
      this.filterParameter$.next(auditIdFilter);
    }
  }

  public get statusFilter(): string {
    return this._statusFilter;
  }

  public set statusFilter(v: string) {
    if (this._statusFilter !== v) {
      this._statusFilter = v;
      const statusFilter = `status#${v}`;
      this.filterParameter$.next(statusFilter);
    }
  }

  public get priorityFilter(): string {
    return this._priorityFilter;
  }

  public set priorityFilter(v: string) {
    if (this._priorityFilter !== v) {
      this._priorityFilter = v;
      const priorityFilter = `priority#${v}`;
      this.filterParameter$.next(priorityFilter);
    }
  }

  public get dueFromDateFilter(): Date | undefined {
    return this._fromDueDateFilter;
  }

  public set dueFromDateFilter(v: Date | undefined) {
    if (this._fromDueDateFilter !== v) {
      this._fromDueDateFilter = v;
      const fromDateFilter = `fromDate#${v}`;
      this.filterParameter$.next(fromDateFilter);
    }
  }

  public get dueToDateFilter(): Date | undefined {
    return this._toDueDateFilter;
  }

  public set dueToDateFilter(v: Date | undefined) {
    if (this._toDueDateFilter !== v) {
      this._toDueDateFilter = v;
      const toDateFilter = `toDate#${v}`;
      this.filterParameter$.next(toDateFilter);
    }
  }

  ngOnInit(): void {
    this.getAllAudits();
    this.route.queryParamMap.subscribe((queryParams) => {
      const auditId = queryParams.get('auditId');
      if (auditId) {
        const auditCheckInterval = setInterval(() => {
          if (this.auditList.length > 0) {
            clearInterval(auditCheckInterval);
            this.auditIdFilter = auditId;
            this.capaRequestFilterParameter.auditId = auditId;
          }
        }, 100);
      }
    });
    this.sub$.sink = this.filterParameter$
      .pipe(debounceTime(1000), distinctUntilChanged())
      .subscribe((c: string) => {
        this.capaRequestFilterParameter.skip = 0;
        this.paginator.pageIndex = 0;
        const filterArray: Array<string> = c.split('#');
        if (filterArray[0] === 'referenceNo') {
          this.capaRequestFilterParameter.referenceNo = filterArray[1];
        } else if (filterArray[0] === 'title') {
          this.capaRequestFilterParameter.title = filterArray[1];
        } else if (filterArray[0] === 'raisedById') {
          this.capaRequestFilterParameter.raisedById = filterArray[1];
        } else if (filterArray[0] === 'auditId') {
          this.capaRequestFilterParameter.auditId = filterArray[1];
        } else if (filterArray[0] === 'priority') {
          this.capaRequestFilterParameter.priority = filterArray[1];
        } else if (filterArray[0] === 'status') {
          this.capaRequestFilterParameter.capaRequestStatus = filterArray[1];
        } else if (filterArray[0] === 'fromDate') {
          if (filterArray[1] != 'null') {
            this.capaRequestFilterParameter.dueStartDate = new Date(
              filterArray[1]
            );
            this.capaRequestFilterParameter.dueEndDate = this.dueToDateFilter;
          } else {
            this.capaRequestFilterParameter.dueStartDate = undefined;
            this.capaRequestFilterParameter.dueEndDate = undefined;
          }
        } else if (filterArray[0] === 'toDate') {
          if (filterArray[1] != 'null') {
            this.capaRequestFilterParameter.dueEndDate = new Date(
              filterArray[1]
            );
            this.capaRequestFilterParameter.dueStartDate =
              this.dueFromDateFilter;
          } else {
            this.capaRequestFilterParameter.dueStartDate = undefined;
            this.capaRequestFilterParameter.dueEndDate = undefined;
          }
        }
        this.capaRequestStore.loadByQuery(this.capaRequestFilterParameter);
      });
  }

  clearOrderDates() {
    this.dueFromDateFilter = undefined;
    this.dueToDateFilter = undefined;
  }

  getAllAudits() {
    this.capaRquestService.getAllAudits().subscribe({
      next: (result: IdName[] | CommonError) => {
        if (Array.isArray(result)) {
          this.auditList = result;
        } else {
          this.auditList = [];
        }
      },
    });
  }

  deleteCapaRequest(capaRequestId: string) {
    this.commandDialogService
      .deleteConfirmtionDialog(
        this.translationService.getValue(
          'ARE_YOU_SURE_YOU_WANT_TO_DELETE_THIS_CAPA_REQUEST'
        )
      )
      .subscribe({
        next: (result: boolean) => {
          if (result) {
            this.capaRequestStore.deleteCapaRequestById(capaRequestId);
          }
        },
      });
  }

  refresh() {
    this.capaRequestStore.loadByQuery(this.capaRequestFilterParameter);
  }

  ngAfterViewInit(): void {
    this.sub$.sink = this.sort.sortChange.subscribe(() => {
      this.paginator.pageIndex = 0;
    });
    this.sub$.sink = merge(this.sort.sortChange, this.paginator.page)
      .pipe(
        tap(() => {
          this.capaRequestFilterParameter.skip =
            this.paginator.pageIndex * this.paginator.pageSize;
          this.capaRequestFilterParameter.pageSize = this.paginator.pageSize;
          this.capaRequestFilterParameter.orderBy =
            this.sort.active + ' ' + this.sort.direction;
          this.capaRequestStore.loadByQuery(this.capaRequestFilterParameter);
        })
      )
      .subscribe();
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.capaRequestStore.capaRequestList().indexOf(row);
  }
}
