import { inject, Injectable } from '@angular/core';
import { CommonHttpErrorService } from '@core/error-handler/common-http-error.service';
import { HttpClient, HttpParams, HttpResponse } from '@angular/common/http'; // Fixed import
import { Observable } from 'rxjs';
import { catchError } from 'rxjs/operators';
import { CapaRequest } from '../model/capa-request';
import { CommonError } from '@core/error-handler/common-error';
import { CapaRequestResource } from '../model/capa-request-resource';
import { IdName } from '@core/domain-classes/id-name';

@Injectable({
  providedIn: 'root'
})
export class CapaRequestService {
  private httpClient = inject(HttpClient);
  private commonHttpErrorService = inject(CommonHttpErrorService);
  constructor() { }

  getCapaRequests(
    resource: CapaRequestResource
  ): Observable<HttpResponse<CapaRequest[]>> {
    const url = 'CapaRequest';
    const customParams = new HttpParams()
      .set('OrderBy', resource.orderBy)
      .set('PageSize', resource.pageSize)
      .set('Skip', resource.skip)
      .set('ReferenceNo', resource.referenceNo)
      .set('Title', resource.title)
      .set('Priority', resource.priority)
      .set('RaisedById', resource.raisedById)
      .set('AuditId', resource.auditId)
      .set('CapaRequestStatus', resource.capaRequestStatus)
      .set('RaisedById', resource.raisedById)
      .set('DueStartDate', resource.dueStartDate ? resource.dueStartDate.toISOString() : '')
      .set('DueEndDate', resource.dueEndDate ? resource.dueEndDate.toISOString() : '');
    return this.httpClient.get<CapaRequest[]>(url, {
      params: customParams,
      observe: 'response'
    });
  }

  getCapaRequestById(id: string): Observable<CapaRequest | CommonError> {
    const url = `CapaRequest/${id}`;
    return this.httpClient
      .get<CapaRequest>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  addCapaRequest(
    setting: CapaRequest
  ): Observable<CapaRequest | CommonError> {
    const url = `CapaRequest`;
    return this.httpClient
      .post<CapaRequest>(url, setting)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  updateCapaRequest(
    setting: CapaRequest
  ): Observable<CapaRequest | CommonError> {
    const url = `CapaRequest/${setting.id}`;
    return this.httpClient
      .put<CapaRequest>(url, setting)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  deleteCapaRequest(
    id: string
  ): Observable<CapaRequest | CommonError> {
    const url = `CapaRequest/${id}`;
    return this.httpClient
      .delete<CapaRequest>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getAllAudits(): Observable<IdName[] | CommonError> {
    const url = 'CapaRequest/audits';
    return this.httpClient.get<IdName[]>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

}
