import { patchState, signalStore, withHooks, withMethods, withState } from "@ngrx/signals";
import { rxMethod } from "@ngrx/signals/rxjs-interop";
import { inject } from "@angular/core";
import { ToastrService } from '@core/services/toastr-service';
import { TranslationService } from "@core/services/translation.service";
import { distinctUntilChanged, pipe, switchMap, tap } from "rxjs";
import { tapResponse } from "@ngrx/operators";
import { CapaRequest } from "../../capa/model/capa-request";
import { toObservable } from "@angular/core/rxjs-interop";
import { HttpResponse } from "@angular/common/http";
import { CapaRequestService } from "./capa-request.service";
import { CapaRequestResource } from "../model/capa-request-resource";

type CapaRequestState = {
  capaRequestList: CapaRequest[];
  capaRequest: CapaRequest;
  loadList: boolean;
  isAddUpdate: boolean;
  filterParameters: CapaRequestResource;
};

export const initialCapaRequestState: CapaRequestState = {
  capaRequestList: [],
  capaRequest: {} as CapaRequest,
  loadList: true,
  isAddUpdate: false,
  filterParameters: {
    orderBy: 'referenceNo asc',
    pageSize: 10,
    skip: 0,
    totalCount: 0,
    referenceNo: '',
    title: '',
    priority: '',
    capaRequestStatus: '',
    raisedById: '',
    auditId: '',
    dueStartDate: undefined,
    dueEndDate: undefined,
  }
};

export const CapaRequestStore = signalStore(
  { providedIn: 'root' },
  withState(initialCapaRequestState),
  withMethods(
    (
      store,
      capaRequestService = inject(CapaRequestService),
      toastrService = inject(ToastrService),
      translationService = inject(TranslationService),
    ) => ({
      loadByQuery: rxMethod<CapaRequestResource>(
        pipe(
          tap(() => patchState(store, { capaRequestList: [] })),
          switchMap((filter) => {
            return capaRequestService.getCapaRequests(filter).pipe(
              tapResponse({
                next: (httpResponse: HttpResponse<CapaRequest[]>) => {
                  if (httpResponse && httpResponse.body) {
                    patchState(store, { capaRequestList: [...httpResponse.body], loadList: false })
                  }
                  if (httpResponse.headers.get('x-pagination')) {
                    const pagination: CapaRequestResource = JSON.parse(httpResponse.headers.get('x-pagination') ?? '');
                    patchState(store, { filterParameters: { ...filter, totalCount: pagination.totalCount } })
                  }
                },
                error: (error: any) => {
                  toastrService.error(error.error);
                }
              })
            );
          })
        )
      ),
      addCapaRequest: rxMethod<CapaRequest>(
        pipe(
          distinctUntilChanged(),
          tap(() => patchState(store, { loadList: false })),
          switchMap((capaRequest: CapaRequest) =>
            capaRequestService.addCapaRequest(capaRequest).pipe(
              tapResponse({
                next: () => {
                  toastrService.success(
                    translationService.getValue('CAPA_REQUEST_CREATED_SUCCESSFULLY')
                  );
                  patchState(store, { isAddUpdate: true, loadList: true });
                },
                error: (err: any) => {
                  console.error(err);
                },
              })
            )
          )
        )
      ),
      updateCapaRequest: rxMethod<CapaRequest>(
        pipe(
          distinctUntilChanged(),
          tap(() => patchState(store, { loadList: false })),
          switchMap((capaRequest: CapaRequest) =>
            capaRequestService.updateCapaRequest(capaRequest).pipe(
              tapResponse({
                next: (response) => {
                  const updatedCapaRequest: CapaRequest = response as CapaRequest;
                  toastrService.success(
                    translationService.getValue('CAPA_REQUEST_UPDATED_SUCCESSFULLY')
                  );
                  patchState(store, {
                    capaRequestList: store.capaRequestList().map((cr) => {
                      return updatedCapaRequest.id === cr.id ? updatedCapaRequest : cr;
                    }),
                    isAddUpdate: true,
                  });
                },
                error: (err: any) => {
                  console.error(err);
                },
              })
            )
          )
        )
      ),
      deleteCapaRequestById: rxMethod<string>(
        pipe(
          distinctUntilChanged(),
          tap(() => patchState(store, { loadList: false })),
          switchMap((capaRequestId: string) =>
            capaRequestService.deleteCapaRequest(capaRequestId).pipe(
              tapResponse({
                next: () => {
                  toastrService.success(
                    translationService.getValue('CAPA_REQUEST_DELETED_SUCCESSFULLY')
                  );
                  patchState(store, {
                    capaRequestList: store.capaRequestList().filter((w) => w.id !== capaRequestId),
                    filterParameters: {
                      ...store.filterParameters(),
                      totalCount: store.filterParameters().totalCount - 1,
                    },
                  });
                },
                error: (err: any) => {
                  console.error(err);
                },
              })
            )
          )
        )
      ),
      resetflag() {
        patchState(store, { isAddUpdate: false })
      },
    })
  ),
  withHooks({
    onInit(store) {
      toObservable(store.loadList).subscribe((flag) => {
        if (flag) {
          store.loadByQuery(store.filterParameters());
        }
      });
    },
  })
);
