import { Component, Inject, inject, OnInit, Optional } from '@angular/core';
import {
  FormBuilder,
  FormGroup,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { MatCardModule } from '@angular/material/card';
import { BaseComponent } from '../../../base.component';
import { MatButtonModule } from '@angular/material/button';
import { CAPA_REQUEST_STATUS } from '../../model/capa-Request-Status-enum';
import { CAPA_REQUEST_PRIORITY } from '../../model/capa-Request-Priority-enum';
import { MatSelectModule } from '@angular/material/select';
import { CommonService } from '@core/services/common.service';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { CapaRequestStore } from '../capa-request.store';
import { ActivatedRoute, Router, RouterLink } from '@angular/router';
import { CapaRequest } from '../../model/capa-request';
import { toObservable } from '@angular/core/rxjs-interop';
import { NgClass } from '@angular/common';
import { TextEditorComponent } from '@shared/text-editor/text-editor.component';
import { UserStore } from '../../../user/store/user.store';
import { IdName } from '@core/domain-classes/id-name';
import { AuditService } from '../../../audit/audit.service';

import { ToastrService } from '@core/services/toastr-service';
import { CommonError } from '@core/error-handler/common-error';
import { TranslateModule } from '@ngx-translate/core';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { MatIconModule } from '@angular/material/icon';
import { MAT_DIALOG_DATA, MatDialogRef } from '@angular/material/dialog';

@Component({
  selector: 'app-manage-capa-request',
  imports: [
    ReactiveFormsModule,
    MatCardModule,
    MatButtonModule,
    MatSelectModule,
    MatDatepickerModule,
    NgClass,
    TextEditorComponent,
    TranslateModule,
    PageHelpTextComponent,
    MatIconModule
  ],
  templateUrl: './manage-capa-request.component.html',
  styleUrl: './manage-capa-request.component.scss',
})
export class ManageCapaRequestComponent
  extends BaseComponent
  implements OnInit {
  capaRequestForm: FormGroup;
  fb = inject(FormBuilder);
  userStore = inject(UserStore);
  auditService = inject(AuditService);
  capaRequestStore = inject(CapaRequestStore);
  toastrService = inject(ToastrService);
  commonService = inject(CommonService);
  route = inject(ActivatedRoute);
  router = inject(Router);
  capaRequest: CapaRequest;
  isEditMode = false;
  auditList: IdName[] = [];
  isDetailsMode = true;
  isDialog = false;
  ncNumber: string = '';
  minDueDate: Date = new Date();

  statusList = Object.keys(CAPA_REQUEST_STATUS)
    .filter((key) => !isNaN(Number(CAPA_REQUEST_STATUS[key as any])))
    .map((key) => ({
      label: key,
      value: CAPA_REQUEST_STATUS[key as keyof typeof CAPA_REQUEST_STATUS],
    }));
  priorityList = Object.keys(CAPA_REQUEST_PRIORITY)
    .filter((key) => !isNaN(Number(CAPA_REQUEST_PRIORITY[key as any])))
    .map((key) => ({
      name: key,
      id: CAPA_REQUEST_PRIORITY[key as keyof typeof CAPA_REQUEST_PRIORITY],
    }));

  constructor(
    @Optional() private dialogRef: MatDialogRef<ManageCapaRequestComponent>,
    @Optional() @Inject(MAT_DIALOG_DATA) public data: { event: { auditId: string; auditResponseId: string; } },
  ) {
    super();
    this.subscribeIsAddorUpdate();
  }

  ngOnInit(): void {
    this.creatCapaRequestForm();

    if (this.data?.event) {
      this.capaRequestForm.patchValue({
        auditId: this.data.event.auditId,
        auditResponseId: this.data.event.auditResponseId,
      });
      this.capaRequestForm.get('auditId')?.disable();
      this.isDialog = true;
    }

    this.auditService
      .getAuditsForDropdown()
      .subscribe((result: IdName | CommonError) => {
        if (Array.isArray(result)) {
          this.auditList = result;
        } else {
          this.auditList = [];
        }

        this.route.data.subscribe((data) => {
          const capaRequest = data['capaRequestDetail'];
          this.capaRequest = capaRequest;
          this.isEditMode = !!capaRequest;

          if (this.isEditMode && capaRequest) {
            this.capaRequestForm.patchValue({
              ...capaRequest,
              auditId: capaRequest.auditId || '',
            });
            this.capaRequestForm.get('auditId')?.disable();
            this.capaRequestForm.get('description')?.disable();
          } else {
            if (this.data.event) {
              const matchingAudit = this.auditList.find(
                (a) => a.id === this.data.event.auditId
              );
              if (!matchingAudit) {
                this.toastrService.error(
                  this.translationService.getValue('AUDIT_NOT_FOUND')
                );
              } else {
                this.capaRequestForm.patchValue({
                  auditId: matchingAudit.id,
                });
                this.capaRequestForm.get('auditId')?.disable();
              }
            }
            if (this.data.event) {
              this.capaRequestForm.patchValue({
                auditResponseId: this.data.event.auditResponseId,
              });
            }
          }

          this.route.queryParams.subscribe((params) => {
            if (params['NonConformanceId']) {
              this.capaRequestForm.patchValue({
                nonConformanceId: params['NonConformanceId'],
              });
            }

            if (params['NcNumber']) {
              this.ncNumber = params['NcNumber'];
            }

            if (params['isDetailsMode'] === 'true') {
              this.isDetailsMode = false;
              this.isDialog = true;
            }
          });
        });
      });
  }

  creatCapaRequestForm() {
    this.capaRequestForm = this.fb.group({
      id: [''],
      referenceNo: [''],
      title: ['', [Validators.required]],
      raisedById: ['', [Validators.required]],
      description: [''],
      capaRequestStatus: ['', [Validators.required]],
      priority: ['', [Validators.required]],
      dueDate: [''],
      auditId: [''],
      auditResponseId: [''],
      nonConformanceId: [''],
    });
  }

  getAllAuidList() {
    this.sub$.sink = this.auditService
      .getAuditsForDropdown()
      .subscribe((result: IdName) => {
        if (Array.isArray(result)) {
          this.auditList = result;
        } else {
          this.auditList = [];
        }
      });
  }

  onSubmit() {
    if (!this.capaRequestForm.valid) {
      this.capaRequestForm.markAllAsTouched();
      return;
    }

    const capaRequest: CapaRequest = this.capaRequestForm.getRawValue();

    if (this.isEditMode) {
      this.capaRequestStore.updateCapaRequest(capaRequest);
    } else {
      this.capaRequestStore.addCapaRequest(capaRequest);
    }
  }

  subscribeIsAddorUpdate() {
    this.sub$.sink = toObservable(this.capaRequestStore.isAddUpdate).subscribe((flag) => {
      if (!flag) return;

      this.capaRequestStore.resetflag();

      if (this.data !== null) {
        this.dialogRef.close(true);
        return;
      }

      const route = this.isDetailsMode
        ? ['/capa/requests']
        : [`/capa/details/${this.capaRequestForm.get('id')?.value}/review`];
      this.router.navigate(route);
    });
  }

  onCancel() {
    if (this.data !== null) {
      this.dialogRef.close();
      return;
    }
    this.router.navigate(['/capa/requests']);
  }
}
