import { Component, inject, Input, OnChanges, OnInit, SimpleChanges } from '@angular/core';
import { MatTableModule } from '@angular/material/table';
import { BaseComponent } from '../../../base.component';
import { CapaReviewService } from '../capa-review.service';
import { CapaReview } from '../../model/capa-review';
import { MatDialog } from '@angular/material/dialog';
import { ManageCapaReviewComponent } from '../manage-capa-review/manage-capa-review.component';
import { TranslateModule } from '@ngx-translate/core';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { ActivatedRoute } from '@angular/router';
import { CapaReviewStatusPipe } from '../../pipes/capa-review-status.pipe';
import { NgClass } from '@angular/common';
import { LimitToPipe } from '@shared/pipes/limit-to.pipe';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';
import { MatButtonModule } from '@angular/material/button';

@Component({
  selector: 'app-capa-review-list',
  imports: [
    MatTableModule,
    TranslateModule,
    UTCToLocalTime,
    CapaReviewStatusPipe,
    NgClass,
    LimitToPipe,
    MatIconModule,
    MatCardModule,
    MatButtonModule
  ],
  templateUrl: './capa-review-list.component.html',
  styleUrl: './capa-review-list.component.scss'
})
export class CapaReviewListComponent extends BaseComponent implements OnInit, OnChanges {
  @Input() capaReviewResponse: CapaReview | null = null;

  capaReviewList: CapaReview[] = [];
  displayedColumns: string[] = ['action', 'reviewBy', 'status', 'reviewDate'];
  capaReviewService = inject(CapaReviewService);
  dialog = inject(MatDialog);
  route = inject(ActivatedRoute);

  ngOnInit(): void {
    this.route.parent?.paramMap.subscribe(params => {
      const capaRequestId = params.get('id');
      if (capaRequestId) {
        this.getAllReviews(capaRequestId);
      }
    });
  }

  ngOnChanges(changes: SimpleChanges): void {
    if (changes['capaReviewResponse'] && this.capaReviewResponse) {
      if (this.capaReviewResponse.id) {
        this.capaReviewList = [...this.capaReviewList, this.capaReviewResponse];
      }
    }
  }

  getAllReviews(capaRequestId: string) {
    this.capaReviewService.getCapaReviewsById(capaRequestId).subscribe({
      next: (reviews) => {
        const Reviews = reviews as CapaReview[] || [];
        if (Reviews.length > 0) {
          this.capaReviewList = Reviews;
        }
      },
    });
  }

  viewCapaReview(capaReview: CapaReview) {
    const dialogRef = this.dialog.open(ManageCapaReviewComponent, {
      maxWidth: '60vw',
      width: '100%',
      data: { capaReview, isViewOnly: true },
    });

    this.sub$.sink = dialogRef.afterClosed().subscribe((result) => {
      if (result) {
        const index = this.capaReviewList.findIndex(cr => cr.id === result.id);
        if (index !== -1) {
          this.capaReviewList[index] = result;
          this.capaReviewList = [...this.capaReviewList];
        }
      }
    });
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.capaReviewList.indexOf(row);
  }
}
