import { Component, Inject, inject, OnInit, Optional } from '@angular/core';
import { FormBuilder, FormGroup, ReactiveFormsModule, Validators } from '@angular/forms';
import { MatSelectModule } from '@angular/material/select';
import { BaseComponent } from '../../../base.component';
import { CAPA_REVIEW_STATUS } from '../../model/capa-review-status-enum';
import { CapaReview } from '../../model/capa-review';
import { CapaReviewService } from '../capa-review.service';

import { ToastrService } from '@core/services/toastr-service';
import { MatButtonModule } from '@angular/material/button';
import { MatCardModule } from '@angular/material/card';
import { CapaReviewListComponent } from "../capa-review-list/capa-review-list.component";
import { ActivatedRoute, Router, RouterLink } from '@angular/router';
import { TextEditorComponent } from '@shared/text-editor/text-editor.component';
import { NgClass } from '@angular/common';
import { MatDialogRef, MAT_DIALOG_DATA } from '@angular/material/dialog';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { TranslateModule } from '@ngx-translate/core';
import { MatIconModule } from '@angular/material/icon';


@Component({
  selector: 'app-capa-review-manage',
  imports: [
    ReactiveFormsModule,
    MatSelectModule,
    MatButtonModule,
    MatCardModule,
    CapaReviewListComponent,
    TextEditorComponent,
    NgClass,
    HasClaimDirective,
    TranslateModule,
    MatIconModule
  ],
  templateUrl: './manage-capa-review.component.html',
  styleUrl: './manage-capa-review.component.scss'
})
export class ManageCapaReviewComponent extends BaseComponent implements OnInit {
  capaReviewForm: FormGroup;
  fb = inject(FormBuilder);
  capaReviewService = inject(CapaReviewService);
  toastrService = inject(ToastrService);
  route = inject(ActivatedRoute);
  router = inject(Router);

  capaReviewResponse: CapaReview = {} as CapaReview;
  statusList = Object.keys(CAPA_REVIEW_STATUS)
    .filter(key => !isNaN(Number(CAPA_REVIEW_STATUS[key as any])))
    .map(key => ({
      label: key,
      value: CAPA_REVIEW_STATUS[key as keyof typeof CAPA_REVIEW_STATUS]
    }));

  constructor(
    @Optional() private dialogRef: MatDialogRef<ManageCapaReviewComponent>,
    @Optional() @Inject(MAT_DIALOG_DATA) public data: { capaReview: CapaReview, isViewOnly: boolean },
  ) {
    super();
  }

  ngOnInit(): void {
    this.creatCapaReviewForm();

    this.sub$.sink = this.route.parent?.paramMap.subscribe(params => {
      const capaRequestId = params.get('id');
      if (capaRequestId) {
        this.capaReviewForm.patchValue({ capaRequestId: capaRequestId });
      }
    });

    if (this.data !== null) {
      this.capaReviewResponse = this.data.capaReview;
      this.capaReviewForm.patchValue({
        id: this.capaReviewResponse.id,
        capaRequestId: this.capaReviewResponse.capaRequestId,
        status: this.capaReviewResponse.status,
        reviewComment: this.capaReviewResponse.reviewComment,
        reviewByName: this.capaReviewResponse.reviewByName,
        reviewedDate: this.capaReviewResponse.reviewedDate ? new Date(this.capaReviewResponse.reviewedDate).toLocaleDateString('en-GB') : '',
      });
      this.capaReviewForm.get('reviewComment')?.disable();
    } else {
      this.data = {
        capaReview: {} as CapaReview,
        isViewOnly: false
      };
    }
  }

  creatCapaReviewForm(): void {
    this.capaReviewForm = this.fb.group({
      id: [''],
      capaRequestId: [''],
      status: ['', [Validators.required]],
      reviewComment: ['', [Validators.required]],
      reviewByName: [''],
      reviewedDate: [''],
    });
  }

  onSubmit(): void {
    if (!this.capaReviewForm.valid) {
      this.capaReviewForm.markAllAsTouched();
      return;
    }

    const capaReview: CapaReview = this.capaReviewForm.getRawValue();

    if (capaReview.capaRequestId && this.data.capaReview.id) {
      this.capaReviewService.updateCapaReview(capaReview).subscribe({
        next: (response) => {
          this.toastrService.success(this.translationService.getValue('CAPA_REVIEW_UPDATED_SUCCESSFULLY'));
          this.dialogRef.close(response);
        },
      })
    } else {
      this.capaReviewService.creatCapaReview(capaReview).subscribe({
        next: (response) => {
          this.toastrService.success(this.translationService.getValue('CAPA_REVIEW_CREATE_SUCCESSFULLY'));
          this.capaReviewResponse = response as CapaReview;
          this.capaReviewForm.get('status')?.reset();
          this.capaReviewForm.get('reviewComment')?.reset();
        },
        error: (error) => {
          this.toastrService.error(this.translationService.getValue(error));
        }
      })
    }
  }

  onCancel(): void {
    if (this.data.isViewOnly) {
      this.dialogRef.close();
    } else {
      const capaRequestId = this.capaReviewForm.get('capaRequestId')?.value;
      if (capaRequestId) {
        this.router.navigate(['/capa/details', capaRequestId], { queryParams: { isDetailsMode: true } });
      }
    }
  }
}
