import { inject } from '@angular/core';
import { tapResponse } from '@ngrx/operators';
import {
  patchState,
  signalStore,
  withComputed,
  withHooks,
  withMethods,
  withState,
} from '@ngrx/signals';
import { rxMethod } from '@ngrx/signals/rxjs-interop';

import { ToastrService } from '@core/services/toastr-service';
import { debounceTime, distinctUntilChanged, pipe, switchMap, tap } from 'rxjs';
import { toObservable } from '@angular/core/rxjs-interop';
import { Router } from '@angular/router';
import { CapaRootCauseMethodology } from '@core/domain-classes/capa-root-cause-methodology';
import { CommonError } from '@core/error-handler/common-error';
import { CapaRootCauseMethodologyService } from './capa-root-cause-methodology.service';
import { SecurityService } from '@core/security/security.service';
import { TranslationService } from '@core/services/translation.service';

type CapaRootCauseMethodologyState = {
  capaRootCauseMethodologys: CapaRootCauseMethodology[];
  capaRootCauseMethodology: CapaRootCauseMethodology | null;
  isLoading: boolean;
  loadList: boolean;
  isAddUpdate: boolean;
  commonError: CommonError | null;
};

export const initialCapaRootCauseMethodologyState: CapaRootCauseMethodologyState = {
  capaRootCauseMethodologys: [],
  capaRootCauseMethodology: null,
  isLoading: false,
  loadList: false,
  isAddUpdate: false,
  commonError: null,
};

export const CapaRootCauseMethodologyStore = signalStore(
  { providedIn: 'root' },
  withState(initialCapaRootCauseMethodologyState),
  withComputed(({ }) => ({})),
  withMethods(
    (
      store,
      capaRootCauseMethodologyService = inject(CapaRootCauseMethodologyService),
      toastrService = inject(ToastrService),
      translationService = inject(TranslationService),
    ) => ({
      loadCapaRootCauseMethodologys: rxMethod<void>(
        pipe(
          debounceTime(300),
          tap(() => patchState(store, { isLoading: true })),
          switchMap(() =>
            capaRootCauseMethodologyService.getCapaRootCauseMethodologys().pipe(
              tapResponse({
                next: (capaRootCauseMethodologys: CapaRootCauseMethodology[] | CommonError) => {
                  const capaRootCauseMethodologysNew = capaRootCauseMethodologys as CapaRootCauseMethodology[];
                  patchState(store, {
                    capaRootCauseMethodologys: [...capaRootCauseMethodologysNew],
                    isLoading: false,
                    commonError: null,
                    loadList: false
                  });
                },
                error: (err: CommonError) => {
                  patchState(store, { commonError: err, isLoading: false, loadList: false });
                  console.error(err);
                },
              })
            )
          )
        )
      ),
      deleteCapaRootCauseMethodologyById: rxMethod<string>(
        pipe(
          distinctUntilChanged(),
          tap(() => patchState(store, { isLoading: true })),
          switchMap((capaRootCauseMethodologyId: string) =>
            capaRootCauseMethodologyService.deleteCapaRootCauseMethodology(capaRootCauseMethodologyId).pipe(
              tapResponse({
                next: () => {
                  toastrService.success(
                    translationService.getValue('CAPA_ROOT_CAUSE_METHODOLOGY_DELETED_SUCCESSFULLY')
                  );
                  patchState(store, {
                    capaRootCauseMethodologys: store.capaRootCauseMethodologys().filter((w) => w.id !== capaRootCauseMethodologyId),
                    isLoading: false,
                  });
                },
                error: (err: CommonError) => {
                  patchState(store, { commonError: err, isLoading: false });
                  console.error(err);
                },
              })
            )
          )
        )
      ),
      addUpdateCapaRootCauseMethodology: rxMethod<CapaRootCauseMethodology>(
        pipe(
          distinctUntilChanged(),
          tap(() => patchState(store, { isLoading: true })),
          switchMap((capaRootCauseMethodology: CapaRootCauseMethodology) => {
            if (capaRootCauseMethodology.id) {
              return capaRootCauseMethodologyService.updateCapaRootCauseMethodology(capaRootCauseMethodology).pipe(
                tapResponse({
                  next: () => {
                    toastrService.success(
                      translationService.getValue('CAPA_ROOT_CAUSE_METHODOLOGY_UPDATED_SUCCESSFULLY')
                    );
                    patchState(store, {
                      isLoading: false,
                      isAddUpdate: true,
                      capaRootCauseMethodologys: store.capaRootCauseMethodologys().map((w) =>
                        w.id === capaRootCauseMethodology.id ? capaRootCauseMethodology : w
                      ),
                      capaRootCauseMethodology: capaRootCauseMethodology
                    });
                  },
                  error: (err: CommonError) => {
                    patchState(store, { commonError: err, isLoading: false });
                  },
                })
              );
            } else {
              return capaRootCauseMethodologyService.addCapaRootCauseMethodology(capaRootCauseMethodology).pipe(
                tapResponse({
                  next: (response) => {
                    const capaRootCauseMethodology = response as CapaRootCauseMethodology;
                    toastrService.success(
                      translationService.getValue('CAPA_ROOT_CAUSE_METHODOLOGY_CREATED_SUCCESSFULLY')
                    );
                    patchState(store, {
                      isLoading: false,
                      loadList: true,
                      isAddUpdate: true,
                      capaRootCauseMethodology: capaRootCauseMethodology
                    });
                  },
                  error: (err: CommonError) => {
                    patchState(store, { commonError: err, isLoading: false });
                  },
                })
              );
            }
          })
        )
      ),
      getCapaRootCauseMethodologyById: rxMethod<string>(
        pipe(
          tap(() => patchState(store, { isLoading: true })),
          switchMap((documentMetaTagId: string) =>
            capaRootCauseMethodologyService.getCapaRootCauseMethodology(documentMetaTagId).pipe(
              tapResponse({
                next: (capaRootCauseMethodology: CapaRootCauseMethodology | CommonError) => {
                  const capaRootCauseMethodologyNew = capaRootCauseMethodology as CapaRootCauseMethodology;
                  patchState(store, {
                    capaRootCauseMethodology: capaRootCauseMethodologyNew,
                    isLoading: false,
                    commonError: null,
                  });
                },
                error: (err: CommonError) => {
                  patchState(store, { commonError: err, isLoading: false });
                  console.error(err);
                },
              })
            )
          )
        )
      ),
      resetflag() {
        patchState(store, { isAddUpdate: false });
      }
    })
  ),
  withHooks({
    onInit(store, securityService = inject(SecurityService)) {
      toObservable(store.loadList).subscribe((flag) => {
        if (flag) {
          store.loadCapaRootCauseMethodologys();
        }
      });
      if (
        securityService.isUserAuthenticate()
      ) {
        store.loadCapaRootCauseMethodologys();
      }
    },
  })
);
