import { Component, Inject, inject, OnInit, Optional } from '@angular/core';
import { FormGroup, FormBuilder, Validators, ReactiveFormsModule } from '@angular/forms';
import { MatButtonModule } from '@angular/material/button';
import { MatCardModule } from '@angular/material/card';
import { MatSelectModule } from '@angular/material/select';
import { RouterLink, ActivatedRoute } from '@angular/router';

import { ToastrService } from '@core/services/toastr-service';
import { CapaRootCauseService } from '../capa-root-cause.service';
import { CapaRootCause } from '../../model/capa-root-cause';
import { BaseComponent } from '../../../base.component';
import { CapaRootCauseMethodology } from '@core/domain-classes/capa-root-cause-methodology';
import { CapaRootCauseMethodologyStore } from '../../capa-root-cause-methodology/capa-root-cause-methodology-store';
import { CapaRootCauseListComponent } from "../capa-root-cause-list/capa-root-cause-list.component";
import { TextEditorComponent } from "../../../shared/text-editor/text-editor.component";
import { MatDialogRef, MAT_DIALOG_DATA, MatDialog } from '@angular/material/dialog';
import { NgClass } from '@angular/common';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { TranslateModule } from '@ngx-translate/core';
import { MatIconModule } from '@angular/material/icon';
import { ManageCapaRootCauseMethodologyComponent } from '../../capa-root-cause-methodology/manage-capa-root-cause-methodology/manage-capa-root-cause-methodology.component';
import { RichTextRequired } from '@shared/validators/rich-text.validator';

@Component({
  selector: 'app-manage-capa-root-cause',
  imports: [
    MatCardModule,
    ReactiveFormsModule,
    MatSelectModule,
    MatButtonModule,
    MatIconModule,
    RouterLink,
    CapaRootCauseListComponent,
    TextEditorComponent,
    NgClass,
    HasClaimDirective,
    TranslateModule
  ],
  templateUrl: './manage-capa-root-cause.component.html',
  styleUrl: './manage-capa-root-cause.component.scss'
})
export class ManageCapaRootCauseComponent extends BaseComponent implements OnInit {
  capaRootCauseForm: FormGroup;
  fb = inject(FormBuilder);
  capaRootCauseService = inject(CapaRootCauseService);
  toastrService = inject(ToastrService);
  router = inject(ActivatedRoute);
  capaRootCauseResponse: CapaRootCause = {} as CapaRootCause;
  capaMethodologyStore = inject(CapaRootCauseMethodologyStore);
  capaMethodologyList: CapaRootCauseMethodology[] = [];
  dialog = inject(MatDialog);

  constructor(
    @Optional() private dialogRef: MatDialogRef<ManageCapaRootCauseComponent>,
    @Optional() @Inject(MAT_DIALOG_DATA) public data: { capaRootCause: CapaRootCause, isViewOnly: boolean },
  ) {
    super();
  }

  ngOnInit(): void {
    this.createCapaRootCauseForm();

    this.router.parent?.paramMap.subscribe(params => {
      const capaRequestId = params.get('id');
      if (capaRequestId) {
        this.capaRootCauseForm.patchValue({ capaRequestId: capaRequestId });
      }
    });

    if (this.data !== null) {
      this.capaRootCauseForm.patchValue({
        methodologyId: this.data.capaRootCause.methodologyId,
        description: this.data.capaRootCause.description,
        createdDate: this.data.capaRootCause.createdDate ? new Date(this.data.capaRootCause.createdDate).toLocaleDateString() : '',
        createdByName: this.data.capaRootCause.createdByName,
      });
    } else {
      this.data = {
        capaRootCause: {} as CapaRootCause,
        isViewOnly: false
      }
    }
  }

  createCapaRootCauseForm(): void {
    this.capaRootCauseForm = this.fb.group({
      id: [''],
      capaRequestId: [''],
      methodologyId: ['', [Validators.required]],
      description: ['', [RichTextRequired]],
      createdDate: [''],
      createdByName: [''],
    });
  }

  onSubmit(): void {
    if (!this.capaRootCauseForm.valid) {
      this.capaRootCauseForm.markAllAsTouched();
      return;
    }

    const capaRootCause: CapaRootCause = this.capaRootCauseForm.getRawValue();

    if (capaRootCause.capaRequestId) {
      this.capaRootCauseService.createCapaRootCause(capaRootCause).subscribe({
        next: (response: CapaRootCause) => {
          this.toastrService.success(this.translationService.getValue('CAPA_ROOT_CAUSE_CREATE_SUCCESSFULLY'));
          this.capaRootCauseResponse = response;
          this.capaRootCauseForm.get('methodologyId')?.reset();
          this.capaRootCauseForm.get('description')?.reset();
        },
        error: (error: any) => {
          this.toastrService.error(this.translationService.getValue(error));
        }
      });
    }
  }

  onCancel(): void {
    this.dialogRef.close();
  }

  addMethodology(): void {
    const dialogRef = this.dialog.open(ManageCapaRootCauseMethodologyComponent, {
      maxWidth: '60vw',
      width: '100%',
    });
    dialogRef.afterClosed().subscribe(result => {
      if (result) {
        this.capaRootCauseForm.get('methodologyId')?.setValue(result.id);
      }
    });
  }
}
