import { Component, inject, Input, OnChanges, OnInit, SimpleChanges } from '@angular/core';
import { ComplaintAction } from '../../model/complaint-action';
import { MatTableModule } from '@angular/material/table';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { BaseComponent } from '../../../base.component';
import { ComplaintActionService } from '../complaint-action.service';

import { ToastrService } from '@core/services/toastr-service';
import { MatDialog } from '@angular/material/dialog';
import { ManageComplaintActionComponent } from '../manage-complaint-action/manage-complaint-action.component';
import { ActivatedRoute } from '@angular/router';
import { TranslateModule } from '@ngx-translate/core';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { ComplaintStatusPipe } from '../../pipes/complaint-status.pipe';
import { NgClass } from '@angular/common';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';
import { MatButtonModule } from '@angular/material/button';

@Component({
  selector: 'app-complaint-action-list',
  imports: [
    MatTableModule,
    TranslateModule,
    UTCToLocalTime,
    HasClaimDirective,
    ComplaintStatusPipe,
    NgClass,
    MatIconModule,
    MatCardModule,
    MatButtonModule
  ],
  templateUrl: './complaint-action-list.component.html',
  styleUrl: './complaint-action-list.component.scss'
})
export class ComplaintActionListComponent extends BaseComponent implements OnInit, OnChanges {
  @Input() complaintResponse: ComplaintAction = {} as ComplaintAction;

  displayedColumns: string[] = ['action', 'actionTitle', 'actionDate', 'responsibleId', 'status', 'dueDate', 'completedDate'];
  complaintActionList: ComplaintAction[] = [];
  complaintActionService = inject(ComplaintActionService);
  commandDialogService = inject(CommonDialogService);
  toastrService = inject(ToastrService);
  dialog = inject(MatDialog);
  route = inject(ActivatedRoute);

  ngOnInit(): void {
    this.route.parent?.paramMap.subscribe(params => {
      const complaintId = params.get('id') || '';
      if (complaintId) {
        this.getAllActions(complaintId);
      }
    });
  }

  getAllActions(complaintId: string) {
    this.sub$.sink = this.complaintActionService.getComplaintActionsByRequestId(complaintId).subscribe({
      next: (actions) => {
        const complaintActions = actions as ComplaintAction[];
        if (complaintActions) {
          this.complaintActionList = complaintActions;
        }
      },
    });
  }

  ngOnChanges(changes: SimpleChanges): void {
    if (changes['complaintResponse'] && this.complaintResponse) {
      this.complaintActionList = [...this.complaintActionList, this.complaintResponse];
    }
  }

  viewComplaintAction(complaintAction: ComplaintAction) {
    const dialogRef = this.dialog.open(ManageComplaintActionComponent, {
      maxWidth: '60vw',
      width: '100%',
      data: { complaintAction, isViewOnly: true },
    });

    this.sub$.sink = dialogRef.afterClosed().subscribe((result: ComplaintAction) => {
      if (result) {
        const index = this.complaintActionList.findIndex(a => a.id === result.id);
        if (index !== -1) {
          this.complaintActionList[index] = result;
          this.complaintActionList = [...this.complaintActionList];
        }
      }
    });
  }

  deleteComplaintAction(capaActionId: string) {
    this.commandDialogService.deleteConfirmtionDialog(this.translationService.getValue("ARE_YOU_SURE_YOU_WANT_TO_DELETE_THIS_COMPLAINT_ACTION")).subscribe({
      next: (result: boolean) => {
        if (result) {
          this.sub$.sink = this.complaintActionService.deleteComplaintAction(capaActionId).subscribe({
            next: () => {
              this.toastrService.success(this.translationService.getValue('CAPA_ACTION_DELETE_SUCCESSFULLY'));
              this.complaintActionList = this.complaintActionList.filter(action => action.id !== capaActionId);
            },
            error: (error) => {
              this.toastrService.error(error);
            }
          });
        }
      }
    });
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.complaintActionList.indexOf(row);
  }
}
