import { Component, Inject, inject, OnInit, Optional } from '@angular/core';
import { ActivatedRoute, Router } from '@angular/router';
import { ComplaintActionListComponent } from "../complaint-action-list/complaint-action-list.component";
import { FormBuilder, FormGroup, ReactiveFormsModule, Validators } from '@angular/forms';
import { MatCardModule } from '@angular/material/card';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { MatSelectModule } from '@angular/material/select';
import { UserStore } from '../../../user/store/user.store';
import { BaseComponent } from '../../../base.component';
import { COMPLAINT_STATUS } from '../../model/complaint-status.enum';
import { TextEditorComponent } from '@shared/text-editor/text-editor.component';
import { MatButtonModule } from '@angular/material/button';
import { ComplaintActionService } from '../complaint-action.service';
import { ComplaintAction } from '../../model/complaint-action';

import { ToastrService } from '@core/services/toastr-service';
import { TranslationService } from '@core/services/translation.service';
import { MAT_DIALOG_DATA, MatDialogRef } from '@angular/material/dialog';
import { NgClass } from '@angular/common';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { TranslateModule } from '@ngx-translate/core';
import { MatIconModule } from '@angular/material/icon';

@Component({
  selector: 'app-manage-complaint-action',
  imports: [
    ComplaintActionListComponent,
    ReactiveFormsModule,
    MatCardModule,
    MatDatepickerModule,
    MatSelectModule,
    TextEditorComponent,
    MatButtonModule,
    NgClass,
    HasClaimDirective,
    TranslateModule,
    MatIconModule
  ],
  templateUrl: './manage-complaint-action.component.html',
  styleUrl: './manage-complaint-action.component.scss'
})
export class ManageComplaintActionComponent extends BaseComponent implements OnInit {
  complaintActionForm: FormGroup;
  fb = inject(FormBuilder);
  minDueDate: Date = new Date();
  userStore = inject(UserStore);
  complaintActionResponse: ComplaintAction = {} as ComplaintAction;
  route = inject(ActivatedRoute);
  complaintService = inject(ComplaintActionService);
  toastrService = inject(ToastrService);
  router = inject(Router);

  statusList = Object.keys(COMPLAINT_STATUS)
    .filter(key => !isNaN(Number(COMPLAINT_STATUS[key as any])))
    .map(key => ({
      name: key,
      id: COMPLAINT_STATUS[key as keyof typeof COMPLAINT_STATUS]
    }));

  constructor(
    @Optional() private dialogRef: MatDialogRef<ManageComplaintActionComponent>,
    @Optional() @Inject(MAT_DIALOG_DATA) public data: { complaintAction: ComplaintAction, isViewOnly: boolean },
  ) {
    super();
  }

  ngOnInit(): void {
    this.createComplaintActionForm();

    this.route.parent?.paramMap.subscribe(params => {
      const complaintId = params.get('id') || '';
      if (complaintId) {
        this.complaintActionForm.get('complaintId')?.setValue(complaintId);
      }
    });

    if (this.data !== null) {
      this.complaintActionForm.patchValue(this.data.complaintAction);
      if (this.data.isViewOnly) {
        this.complaintActionForm.get('actionDetail')?.disable();
      }
    } else {
      this.data = {
        complaintAction: {} as ComplaintAction,
        isViewOnly: false
      }
    }
  }

  createComplaintActionForm() {
    this.complaintActionForm = this.fb.group({
      id: [''],
      complaintId: [''],
      actionTitle: ['', [Validators.required]],
      actionDetail: ['', [Validators.required]],
      actionDate: [new Date(), [Validators.required]],
      responsibleId: ['', [Validators.required]],
      status: [null, [Validators.required]],
      dueDate: [null],
      completedDate: [null],
    });
  }

  onSubmit() {
    if (this.complaintActionForm.invalid) {
      this.complaintActionForm.markAllAsTouched();
      return;
    }

    const complaintAction: ComplaintAction = this.complaintActionForm.getRawValue();
    const complaintId = complaintAction.complaintId;

    if (!complaintId) {
      this.toastrService.error(this.translationService.getValue('COMPLAINT_ID_IS_REQUIRED'));
      return;
    }

    if (complaintAction.id) {
      this.sub$.sink = this.complaintService.updateComplaintAction(complaintAction).subscribe({
        next: (response) => {
          const complaintAction = response as ComplaintAction;
          if (complaintAction) {
            this.toastrService.success(this.translationService.getValue('COMPLAINT_ACTION_UPDATED_SUCCESSFULLY'));
            this.complaintActionForm.get('complaintId')?.setValue(complaintAction.complaintId);
            this.dialogRef.close(complaintAction);
          }
        }
      });
    } else {
      this.sub$.sink = this.complaintService.addComplaintAction(complaintAction).subscribe({
        next: (response) => {
          const complaint = response as ComplaintAction;
          if (complaint) {
            this.toastrService.success(this.translationService.getValue('COMPLAINT_ACTION_CREATED_SUCCESSFULLY'));
            this.complaintActionResponse = complaint;
            this.complaintActionForm.reset();
            this.complaintActionForm.get('complaintId')?.setValue(complaint.complaintId);
          }
        }
      });
    }
  }

  onCancel() {
    if (this.data.isViewOnly) {
      this.dialogRef.close();
    } else {
      const complaintId = this.complaintActionForm.get('complaintId')?.value;
      if (complaintId) {
        this.router.navigate(['/complaint/details', complaintId], { queryParams: { isDetailsMode: true } });
      }
    }
  }
}
