import { NgClass, CommonModule } from '@angular/common';
import { MatExpansionModule } from '@angular/material/expansion';
import { Component, inject, OnInit } from '@angular/core';
import { FormsModule } from '@angular/forms';
import { MatButtonModule } from '@angular/material/button';
import { MatCardModule } from '@angular/material/card';
import { MatIconModule } from '@angular/material/icon';
import { MatTooltipModule } from '@angular/material/tooltip';
import { ToastrService } from '@core/services/toastr-service';
import { TranslationService } from '@core/services/translation.service';
import { TranslateModule } from '@ngx-translate/core';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { TextEditorComponent } from '@shared/text-editor/text-editor.component';
import { BaseComponent } from '../../../base.component';
import { ComplaintActionService } from '../complaint-action.service';
import { ComplaintStatusPipe } from '../../pipes/complaint-status.pipe';
import { ComplaintPriorityPipe } from '../../pipes/complaint-priority.pipe';
import { ComplaintActionStatusPipe } from '../../pipes/complaint-action-status.pipe';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { COMPLAINT_PRIORITY } from '../../model/complaint-priority.enum';
import { COMPLAINT_STATUS } from '../../model/complaint-status.enum';
import { COMPLAINT_ACTION_STATUS } from '../../model/complaint-action-status-enum';
import { Complaint } from '../../model/complaint';
import { ComplaintAction } from '../../model/complaint-action';
import { ComplaintInvestigation } from '../../model/complaint-investigation';
import { ComplaintInvestigationService } from '../../complaint-investigation/complaint-investigation.service';

@Component({
  selector: 'app-my-complaint-action-list',
  imports: [
    MatCardModule,
    MatButtonModule,
    MatIconModule,
    MatTooltipModule,
    NgClass,
    CommonModule,
    MatExpansionModule,
    TranslateModule,
    UTCToLocalTime,
    ComplaintActionStatusPipe,
    ComplaintStatusPipe,
    ComplaintPriorityPipe,
    PageHelpTextComponent,
    FormsModule,
    TextEditorComponent
  ],
  templateUrl: './my-complaint-list.html',
  styleUrl: './my-complaint-list.scss'
})
export class MyComplaintActionList extends BaseComponent implements OnInit {
  complaintRequests: Complaint[] = [];
  expandedIndex: number | null = null;
  COMPLAINT_PRIORITY = COMPLAINT_PRIORITY;
  COMPLAINT_STATUS = COMPLAINT_STATUS;
  COMPLAINT_ACTION_STATUS = COMPLAINT_ACTION_STATUS;

  private complaintActionService = inject(ComplaintActionService);
  private toastrService = inject(ToastrService);
  private translateService = inject(TranslationService);
  private commonDialogService = inject(CommonDialogService);
  private complaintInvestigationService = inject(ComplaintInvestigationService);

  ngOnInit(): void {
    this.loadUserComplaintActions();
  }

  loadUserComplaintActions(): void {
    this.sub$.sink = this.complaintActionService.getUserComplaintActions().subscribe({
      next: (actions: Complaint[] | any) => {
        const complaintActions = actions as Complaint[];
        if (complaintActions) {
          this.complaintRequests = complaintActions;
        }
      }
    });
  }

  updateActionStatus(action: ComplaintAction, newStatus: number): void {
    const complaintAction: ComplaintAction = { ...action, status: newStatus };

    this.sub$.sink = this.commonDialogService
      .deleteConfirmtionDialog(`${this.translateService.getValue('ARE_YOU_SURE_YOU_WANT_TO_UPDATE_STATUS')}`)
      .subscribe((isTrue: boolean) => {
        if (isTrue) {
          this.complaintActionService.updateComplaintAction(complaintAction).subscribe({
            next: () => {
              action.status = newStatus;
              this.toastrService.success(this.translateService.getValue('COMPLAINT_ACTION_UPDATED_SUCCESSFULLY'));
            }
          });
        }
      });
  }

  updateInvestigation(complaint: ComplaintInvestigation) {
    const complaintInvestigation = { ...complaint };

    this.sub$.sink = this.complaintInvestigationService.updateComplaintInvestigation(complaintInvestigation).subscribe({
      next: () => {
        this.toastrService.success(this.translateService.getValue('COMPLAINT_INVESTIGATION_UPDATED_SUCCESSFULLY'));
      }
    });
  }

  onAccordionToggle(index: number) {
    this.expandedIndex = this.expandedIndex === index ? null : index;
  }
}
