import { HttpClient } from '@angular/common/http';
import { inject, Injectable } from '@angular/core';
import { CommonError } from '@core/error-handler/common-error';
import { CommonHttpErrorService } from '@core/error-handler/common-http-error.service';
import { catchError, Observable, retry, timer } from 'rxjs';
import { ComplaintAttachment } from '../model/complaint-attachment';
import { ComplaintAttachmentChunk } from '../model/complaint-attachment-chunk';

@Injectable({
  providedIn: 'root'
})
export class ComplaintAttachmentService {
  maxRetries: number = 2; // Maximum number of retries for chunk uploads
  private httpClient = inject(HttpClient);
  private commonHttpErrorService = inject(CommonHttpErrorService);

  saveComplaintAttachment(response: ComplaintAttachment): Observable<ComplaintAttachment | CommonError> {
    const url = 'ComplaintAttachment';
    return this.httpClient
      .post<ComplaintAttachment>(url, response)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  uploadComplaintAttachmentChunk(complaintAttachmentChunk: FormData): Observable<ComplaintAttachmentChunk | CommonError> {
    const url = 'ComplaintAttachment/Chunk';
    return this.httpClient
      .post<ComplaintAttachmentChunk>(url, complaintAttachmentChunk)
      .pipe(
        retry({
          count: this.maxRetries, // Retry up to 2 times
          delay: (error, retryCount) => {
            console.warn(`Retrying chunk... Attempt ${retryCount}`);
            return timer(Math.pow(2, retryCount) * 1000);
          },
        })
      );
  }

  markAsComplaintAttachmentChunkComplete(complaintAttachmentChunkId: string, status: boolean): Observable<void | CommonError> {
    const url = 'ComplaintAttachment/Chunk/uploadStatus';
    return this.httpClient
      .post<void>(url, { complaintAttachmentId: complaintAttachmentChunkId, status: status, })
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getAllComplaintAttachmentsById(requestId: string): Observable<ComplaintAttachment[]> {
    const url = `ComplaintAttachment/${requestId}`;
    return this.httpClient
      .get<ComplaintAttachment[]>(url);

  }

  deleteUploadedFile(complaintAttachmentId: string): Observable<void | CommonError> {
    const url = `ComplaintAttachment/${complaintAttachmentId}`;
    return this.httpClient
      .delete<void>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }
}
