import { ChangeDetectorRef, Component, inject, OnInit } from '@angular/core';
import { ReactiveFormsModule, FormGroup, FormBuilder, Validators, FormArray } from '@angular/forms';
import { MatButtonModule } from '@angular/material/button';
import { MatCardModule } from '@angular/material/card';
import { MatIconModule } from '@angular/material/icon';
import { ActivatedRoute } from '@angular/router';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { AllowFileExtension } from '@core/domain-classes/allow-file-extension';
import { DocumentView } from '@core/domain-classes/document-view';
import { validateFile } from '@core/domain-classes/extension-types';
import { CommonError } from '@core/error-handler/common-error';
import { CommonService } from '@core/services/common.service';
import { environment } from '@environments/environment';
import { BasePreviewComponent } from '@shared/base-preview/base-preview.component';
import { OverlayPanel } from '@shared/overlay-panel/overlay-panel.service';

import { ToastrService } from '@core/services/toastr-service';
import { Observable, from, concatMap, catchError, of, mergeMap, toArray } from 'rxjs';
import { ComplaintAttachmentService } from '../complaint-attachment.service';
import { BaseComponent } from '../../../base.component';
import { ComplaintAttachment } from '../../model/complaint-attachment';
import { ComplaintAttachmentChunk } from '../../model/complaint-attachment-chunk';
import { ComplaintAttachmentChunkStatus } from '../../model/complaint-attachment-status';
import { QmsModuleEnum } from '@core/domain-classes/qms-module-enum';
import { TranslateModule } from '@ngx-translate/core';
import { HasClaimDirective } from '@shared/has-claim.directive';

@Component({
  selector: 'app-manage-complaint-attachment',
  imports: [
    ReactiveFormsModule,
    MatButtonModule,
    MatCardModule,
    MatIconModule,
    TranslateModule,
    HasClaimDirective
  ],
  templateUrl: './manage-complaint-attachment.component.html',
  styleUrl: './manage-complaint-attachment.component.scss'
})
export class ManageComplaintAttachmentComponent extends BaseComponent implements OnInit {
  fileUploadGroup!: FormGroup;
  fb = inject(FormBuilder);
  toastrService = inject(ToastrService);
  cd = inject(ChangeDetectorRef);
  commonService = inject(CommonService);
  route = inject(ActivatedRoute);
  complaintAttachmentService = inject(ComplaintAttachmentService);
  commandDialogService = inject(CommonDialogService);
  extension: string | null = null;
  loading = false;
  counter = 0;
  chunkSize = environment.chunkSize;
  chunkUploads: Observable<any>[] = [];
  fileArray: any[] = [];
  isDragging = false;
  allowFileExtension: AllowFileExtension[] = [];
  overlay = inject(OverlayPanel);

  get fileInputs(): FormArray {
    return (<FormArray>this.fileUploadGroup.get('files')) as FormArray;
  }

  ngOnInit(): void {
    this.createFileUploadForm();
    this.getAllAllowFileExtension()

    this.route.parent?.paramMap.subscribe((params) => {
      const complaintId = params.get('id');
      if (complaintId) {
        this.fileUploadGroup.get('complaintId')?.setValue(complaintId);
        this.getAllComplaintAttachmentsById(complaintId);
      }
    });
  }

  createFileUploadForm(): void {
    this.fileUploadGroup = this.fb.group({
      complaintId: ['', [Validators.required]],
      files: this.fb.array([]),
    });
  }

  getAllComplaintAttachmentsById(requestId: string): void {
    this.sub$.sink = this.complaintAttachmentService
      .getAllComplaintAttachmentsById(requestId)
      .subscribe({
        next: (data: ComplaintAttachment[]) => {
          const attachments = data as ComplaintAttachment[];
          if (attachments) {
            this.fileArray = data.map((file) => ({
              isSuccess: true,
              name: file.fileName,
              extension: file.extension,
              id: file.id,
              totalChunk: file.totalChunk,
            }));
          }
        },
        error: (error) => {
          this.toastrService.error(error)
        }
      });
  }

  getAllAllowFileExtension() {
    this.sub$.sink = this.commonService.allowFileExtension$.subscribe(
      (allowFileExtension: AllowFileExtension[]) => {
        if (allowFileExtension) {
          this.allowFileExtension = allowFileExtension;
        }
      }
    );
  }

  onDragOver(event: DragEvent): void {
    event.preventDefault();
    event.stopPropagation();
    this.isDragging = true;
  }

  onDragLeave(): void {
    this.isDragging = false;
  }

  async onDrop(event: DragEvent) {
    event.preventDefault();
    event.stopPropagation();
    this.isDragging = false;

    if (event.dataTransfer?.files) {
      await this.fileUpload(event.dataTransfer?.files);
    }
  }

  get allowedFileTypes(): string {
    const allowedExtensionscombine = this.allowFileExtension.map(
      (ext) => ext.extension
    );
    return allowedExtensionscombine.length > 0
      ? allowedExtensionscombine.join(',')
      : '';
  }

  async onFileSelected(event: Event) {
    const input = event.target as HTMLInputElement;
    await this.fileUpload(input.files);
    input.value = '';
  }

  fileExtesionValidation(extension: string): boolean {
    const allowTypeExtenstion = this.allowFileExtension.find((c) =>
      c.extensions?.find((ext) => ext.toLowerCase() === extension.toLowerCase())
    );
    return allowTypeExtenstion ? true : false;
  }

  async fileUpload(files: FileList | null): Promise<void> {
    if (!files) return;

    for (let i = 0; i < files.length; i++) {
      if (!(await validateFile(files[i]))) {
        this.toastrService.error(
          this.translationService.getValue(
            'INVALID_EXTENSION_OR_CORRUPT_INVALID_SIGNATURE'
          )
        );
        this.cd.markForCheck();
        continue;
      }
      const file = files[i];
      this.extension = file.name.split('.').pop() ?? '';
      if (this.fileExtesionValidation(this.extension)) {
        this.fileInputs.push(
          this.fb.group({
            fileName: [file.name],
            file: [file],
            name: [file.name, [Validators.required]],
            extension: [this.extension],
            message: [''],
            isSuccess: [false],
            isLoading: [false],
            complaintId: [this.fileUploadGroup.get('complaintId')?.value],
            complaintAttachmentId: [''],
            totalChunk: [Math.ceil(file.size / this.chunkSize)],
          })
        );
      }
    }
  }

  removeLocalFile(index: number): void {
    if (index >= 0 && index < this.fileInputs.length) {
      this.fileInputs.removeAt(index);
    }
  }

  saveFilesAndDocument(): void {
    if (!this.fileInputs.valid) {
      this.fileInputs.markAllAsTouched();
      return;
    }

    const fileInputsArray = this.fileInputs.controls;
    if (fileInputsArray.length === 0) {
      this.loading = false;
      return;
    }

    const concatObservable$: Observable<any>[] = [];

    fileInputsArray.forEach((control) => {
      if (!control.get('isSuccess')?.value) {
        const complaintAttachment: ComplaintAttachment = {
          fileName: control.get('fileName')?.value,
          fileSize: this.formatFileSize(control.get('file')?.value.size),
          extension: control.get('extension')?.value,
          complaintId: control.get('complaintId')?.value,
          totalChunk: control.get('totalChunk')?.value,
        };
        concatObservable$.push(
          this.saveComplaintAttachmentChunk({ ...complaintAttachment })
        );
      }
    });

    if (concatObservable$.length === 0) {
      this.loading = false;
      return;
    }

    this.chunkUploads = [];
    this.counter = 0;

    this.sub$.sink = from(concatObservable$)
      .pipe(
        concatMap((obs) =>
          obs.pipe(
            catchError((err) => of(`${err.error?.[0] || 'Upload failed'}`))
          )
        )
      )
      .subscribe({
        next: (document: ComplaintAttachment) => {
          this.counter++;
          if (typeof document !== 'string') {
            this.fileInputs.at(this.counter - 1).patchValue({
              isLoading: false,
              complaintAttachmentId: document.id,
              isSuccess: true,
            });
            const fileData = this.fileInputs.at(this.counter - 1).get('file')?.value;
            const totalChunk = this.fileInputs.at(this.counter - 1).get('totalChunk')?.value;
            this.chunkUploads.push(this.uploadFile(document.id ?? '', fileData, totalChunk));
          } else {
            this.fileInputs.at(this.counter - 1).patchValue({
              isLoading: false,
              isSuccess: false,
            });
          }
        },
        error: () => {
          if (this.counter < this.fileInputs.length) {
            this.fileInputs.at(this.counter).patchValue({
              isLoading: false,
              complaintAttachmentId: '',
            });
          }
          this.loading = false;
        },
        complete: () => {
          if (this.chunkUploads.length > 0) {
            this.uploadedChunksAllFile();
            this.loading = false;
          }
        },
      });
  }

  uploadFile(complaintAttachmentId: string, file: File, totalChunk: number): Observable<any> {
    const { parallelCalls } = this.commonService.getNetworkSpeed();
    const chunkUploads: Observable<any>[] = [];

    for (let i = 0; i < totalChunk; i++) {
      const start = i * this.chunkSize;
      const end = Math.min(start + this.chunkSize, file.size);
      const chunk = file.slice(start, end);
      const formData = new FormData();
      formData.append('file', chunk);
      formData.append('chunkIndex', i.toString());
      formData.append('size', this.chunkSize.toString());
      formData.append('extension', this.extension ?? '');
      formData.append('complaintAttachmentId', complaintAttachmentId);
      chunkUploads.push(this.uploadChunk(formData));
    }
    return from(chunkUploads).pipe(
      mergeMap((upload) => upload, parallelCalls),
      toArray(),
      concatMap(() => {
        return this.completeUpload(complaintAttachmentId, true);
      }),
      catchError((err) => {
        const documentChunkStatus: ComplaintAttachmentChunkStatus = {
          complaintAttachmentId: complaintAttachmentId,
          status: false,
        };
        return of(documentChunkStatus);
      })
    );
  }

  uploadedChunksAllFile(): void {
    const resultArray: any[] = []
    this.sub$.sink = from(this.chunkUploads)
      .pipe(
        concatMap((obs) =>
          obs.pipe(
            catchError((err) => of(`${err.error[0] || 'Chunk upload failed'}`))
          )
        )
      )
      .subscribe({
        next: (data: ComplaintAttachmentChunkStatus) => {
          this.loading = false;
          if (data.status) {
            const fileIndex = this.fileInputs.controls.findIndex(
              (ctrl) =>
                ctrl.get('complaintAttachmentId')?.value ===
                data.complaintAttachmentId
            );
            if (fileIndex !== -1) {
              const control = this.fileInputs.at(fileIndex);
              resultArray.push({
                isSuccess: true,
                documentId: data.complaintAttachmentId,
                name: control.get('name')?.value,
                totalChunk: control.get('totalChunk')?.value,
              });
              this.toastrService.success(this.translationService.getValue('FILE_UPLOADED_SUCCESSFULLY'));
            }
          } else {
            this.documentChunkRemove(data.complaintAttachmentId);
          }
        },
        error: () => {
          this.loading = false;
        },
        complete: () => {
          this.fileInputs.clear();
          const newFiles = resultArray
            .filter((f) => f.isSuccess)
            .map((file) => ({
              isSuccess: true,
              name: file.name,
              extension: file.name.split('.').pop(),
              id: file.documentId,
              totalChunk: file.totalChunk || 0,
            }));
          this.fileArray = [...this.fileArray, ...newFiles];
          this.cd.detectChanges();
        },
      });
  }

  formatFileSize(bytes: number): number {
    return parseInt((bytes / 1024).toFixed(1));
  }

  documentChunkRemove(documentId: string) {
    this.loading = true;
    this.sub$.sink = this.complaintAttachmentService
      .markAsComplaintAttachmentChunkComplete(documentId, false)
      .subscribe({
        next: (data) => {
          this.loading = false;
        },
        error: (error) => {
          this.loading = false;
        },
      });
  }

  getFileIcon(fileType: string): string {
    if (fileType.startsWith('image/')) {
      return 'image';
    } else if (fileType.includes('pdf')) {
      return 'picture_as_pdf';
    } else if (fileType.includes('spreadsheet') || fileType.includes('excel')) {
      return 'table_chart';
    } else if (fileType.includes('document') || fileType.includes('word')) {
      return 'description';
    } else {
      return 'insert_drive_file';
    }
  }

  deleteUploadedFile(complaintAttachmentId: string, index: number): void {
    this.commandDialogService.deleteConfirmtionDialog(this.translationService.getValue("ARE_YOU_SURE_YOU_WANT_TO_DELETE_THIS_COMPLAINT_ATTACHMENT")).subscribe({
      next: (result: boolean) => {
        if (result) {
          this.sub$.sink = this.complaintAttachmentService.deleteUploadedFile(complaintAttachmentId).subscribe({
            next: () => {
              this.toastrService.success(this.translationService.getValue('FILE_DELETED_SUCCESSFULLY'));
              this.fileArray.splice(index, 1);
            },
            error: (error) => {
              this.toastrService.error(error.massage);
            },
          });
        }
      }
    });
  }

  saveComplaintAttachmentChunk(complaintAttachment: ComplaintAttachment): Observable<ComplaintAttachment | CommonError> {
    return this.complaintAttachmentService.saveComplaintAttachment(complaintAttachment);
  }

  uploadChunk(formData: FormData): Observable<ComplaintAttachmentChunk | CommonError> {
    return this.complaintAttachmentService.uploadComplaintAttachmentChunk(formData);
  }

  completeUpload(complaintAttachmentId: string, status: boolean): Observable<any> {
    return this.complaintAttachmentService.markAsComplaintAttachmentChunkComplete(complaintAttachmentId, status);
  }

  onMeidaView(document: ComplaintAttachment) {
    const documentView: DocumentView = {
      documentId: document.id,
      name: document.name,
      extension: document.extension,
      isVersion: false,
      isFromPublicPreview: false,
      isPreviewDownloadEnabled: true,
      isFileRequestDocument: false,
      totalChunk: document.totalChunk,
      moduleNo: QmsModuleEnum.Complaint,
    };
    this.overlay.open(BasePreviewComponent, {
      position: 'center',
      origin: 'global',
      panelClass: ['file-preview-overlay-container', 'white-background'],
      data: documentView,
    });
  }
}
