import { Component, inject, Input, OnChanges, OnInit, SimpleChanges } from '@angular/core';
import { ComplaintInvestigation } from '../../model/complaint-investigation';
import { BaseComponent } from '../../../base.component';
import { ComplaintInvestigationService } from '../complaint-investigation.service';
import { MatTableModule } from '@angular/material/table';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';

import { ToastrService } from '@core/services/toastr-service';
import { MatCheckboxModule } from '@angular/material/checkbox';
import { MatDialog } from '@angular/material/dialog';
import { ManageComplaintInvestigationComponent } from '../manage-complaint-investigation/manage-complaint-investigation.component';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { TranslateModule } from '@ngx-translate/core';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { ActivatedRoute } from '@angular/router';
import { NgClass } from '@angular/common';
import { ComplaintStatusPipe } from '../../pipes/complaint-status.pipe';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';
import { MatButtonModule } from '@angular/material/button';


@Component({
  selector: 'app-complaint-investigation-list',
  imports: [
    MatTableModule,
    MatIconModule,
    MatCheckboxModule,
    HasClaimDirective,
    TranslateModule,
    UTCToLocalTime,
    NgClass,
    ComplaintStatusPipe,
    MatCardModule,
    MatButtonModule
  ],
  templateUrl: './complaint-investigation-list.component.html',
  styleUrl: './complaint-investigation-list.component.scss'
})
export class ComplaintInvestigationListComponent extends BaseComponent implements OnInit, OnChanges {
  @Input() complaintInvestigationResponse: ComplaintInvestigation = {} as ComplaintInvestigation;

  displayedColumns: string[] = ['action', 'investigatorName', 'status', 'investigationDate', 'followUpDate', 'actionRequired'];
  complaintInvestigationService = inject(ComplaintInvestigationService);
  commandDialogService = inject(CommonDialogService);
  toastrService = inject(ToastrService);
  dialog = inject(MatDialog);
  route = inject(ActivatedRoute);
  complaintInvestigationList: ComplaintInvestigation[] = [];


  ngOnInit(): void {
    this.route.parent?.paramMap.subscribe(params => {
      const complaintId = params.get('id');
      if (complaintId) {
        this.getComplaintInvestigations(complaintId);
      }
    });
  }

  ngOnChanges(changes: SimpleChanges): void {
    if (changes['complaintInvestigationResponse'] && this.complaintInvestigationResponse) {
      this.complaintInvestigationList = [...this.complaintInvestigationList, this.complaintInvestigationResponse];
    }
  }

  getComplaintInvestigations(complaintId: string) {
    this.sub$.sink = this.complaintInvestigationService.getComplaintInvestigationsByComplaintId(complaintId).subscribe({
      next: (response) => {
        const complaintInvestigationResponse = response as ComplaintInvestigation[];
        if (complaintInvestigationResponse) {
          this.complaintInvestigationList = complaintInvestigationResponse;
        }
      },
    })
  }

  viewComplaintInvestigation(complaintInvestigation: ComplaintInvestigation) {
    const dialogRef = this.dialog.open(ManageComplaintInvestigationComponent, {
      maxWidth: '65vw',
      width: '100%',
      data: { complaintInvestigation, isViewOnly: true },
    });

    this.sub$.sink = dialogRef.afterClosed().subscribe((result: ComplaintInvestigation) => {
      if (result) {
        const index = this.complaintInvestigationList.findIndex(a => a.id === result.id);
        if (index !== -1) {
          this.complaintInvestigationList[index] = result;
          this.complaintInvestigationList = [...this.complaintInvestigationList];
        }
      }
    });
  }

  deleteComplaintInvestigation(complaintId: string) {
    this.commandDialogService.deleteConfirmtionDialog(this.translationService.getValue("ARE_YOU_SURE_YOU_WANT_TO_DELETE_THIS_COMPLAINT_INVESTIGATION")).subscribe({
      next: (result: boolean) => {
        if (result) {
          this.sub$.sink = this.complaintInvestigationService.deleteComplaintInvestigation(complaintId).subscribe({
            next: () => {
              this.toastrService.success(this.translationService.getValue('CAPA_ACTION_DELETE_SUCCESSFULLY'));
              this.complaintInvestigationList = this.complaintInvestigationList.filter(action => action.id !== complaintId);
            },
          });
        }
      }
    });
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.complaintInvestigationList.indexOf(row);
  }
}
