import { Component, Inject, inject, OnInit, Optional } from '@angular/core';
import { ActivatedRoute, Router } from '@angular/router';
import { TextEditorComponent } from '@shared/text-editor/text-editor.component';
import { ComplaintInvestigationListComponent } from "../complaint-investigation-list/complaint-investigation-list.component";
import { FormBuilder, FormGroup, ReactiveFormsModule, Validators } from '@angular/forms';
import { MatCardModule } from '@angular/material/card';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { MatSelectModule } from '@angular/material/select';
import { UserStore } from '../../../user/store/user.store';
import { COMPLAINT_STATUS } from '../../model/complaint-status.enum';
import { BaseComponent } from '../../../base.component';
import { ComplaintInvestigation } from '../../model/complaint-investigation';
import { MatCheckboxModule } from '@angular/material/checkbox';
import { ComplaintInvestigationService } from '../complaint-investigation.service';

import { ToastrService } from '@core/services/toastr-service';
import { MAT_DIALOG_DATA, MatDialogRef } from '@angular/material/dialog';
import { NgClass } from '@angular/common';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { TranslateModule } from '@ngx-translate/core';
import { MatButtonModule } from '@angular/material/button';
import { MatIconModule } from '@angular/material/icon';

@Component({
  selector: 'app-manage-complaint-investigation',
  imports: [
    TextEditorComponent,
    MatIconModule,
    ComplaintInvestigationListComponent,
    MatCardModule,
    MatDatepickerModule,
    MatSelectModule,
    ReactiveFormsModule,
    MatCheckboxModule,
    NgClass,
    HasClaimDirective,
    TranslateModule,
    MatButtonModule
  ],
  templateUrl: './manage-complaint-investigation.component.html',
  styleUrl: './manage-complaint-investigation.component.scss'
})
export class ManageComplaintInvestigationComponent extends BaseComponent implements OnInit {
  complaintInvestigationForm: FormGroup;
  fb = inject(FormBuilder);
  minDueDate: Date = new Date();
  userStore = inject(UserStore);
  complaintInvestigationResponse: ComplaintInvestigation = {} as ComplaintInvestigation;
  complaintInvestigationService = inject(ComplaintInvestigationService);
  toastrService = inject(ToastrService);
  route = inject(ActivatedRoute);
  router = inject(Router);

  statusList = Object.keys(COMPLAINT_STATUS)
    .filter(key => !isNaN(Number(COMPLAINT_STATUS[key as any])))
    .map(key => ({
      name: key,
      id: COMPLAINT_STATUS[key as keyof typeof COMPLAINT_STATUS]
    }));

  constructor(
    @Optional() private dialogRef: MatDialogRef<ManageComplaintInvestigationComponent>,
    @Optional() @Inject(MAT_DIALOG_DATA) public data: { complaintInvestigation: ComplaintInvestigation, isViewOnly: boolean },
  ) {
    super();
  }

  ngOnInit(): void {
    this.createComplaintInvestigationForm();

    this.route.parent?.paramMap.subscribe(params => {
      const complaintId = params.get('id');
      if (complaintId) {
        this.complaintInvestigationForm.get('complaintId')?.setValue(complaintId);
      }
    });

    if (this.data !== null) {
      this.complaintInvestigationForm.patchValue(this.data.complaintInvestigation);
    } else {
      this.data = {
        complaintInvestigation: {} as ComplaintInvestigation,
        isViewOnly: false
      }
    }
  }

  createComplaintInvestigationForm(): void {
    this.complaintInvestigationForm = this.fb.group({
      id: [''],
      complaintId: [''],
      investigatorId: ['', [Validators.required]],
      investigationDate: [null, [Validators.required]],
      followUpDate: [null],
      status: [null, [Validators.required]],
      findings: ['', [Validators.required]],
      recommendations: ['', [Validators.required]],
      actionRequired: [false]
    });
  }

  onSubmit(): void {
    if (!this.complaintInvestigationForm.valid) {
      this.complaintInvestigationForm.markAllAsTouched();
      return;
    }

    const formValue = this.complaintInvestigationForm.getRawValue();

    if (this.data.complaintInvestigation.id) {
      this.sub$.sink = this.complaintInvestigationService.updateComplaintInvestigation(formValue).subscribe({
        next: (response) => {
          const updatedComplaintInvestigation = response as ComplaintInvestigation;
          if (updatedComplaintInvestigation) {
            this.complaintInvestigationResponse = updatedComplaintInvestigation;
            this.toastrService.success(this.translationService.getValue('COMPLAINT_INVESTIGATION_UPDATED_SUCCESSFULLY'));
            this.dialogRef.close(updatedComplaintInvestigation);
          }
        }
      });
    } else {
      this.sub$.sink = this.complaintInvestigationService.addComplaintInvestigation(formValue).subscribe({
        next: (response: ComplaintInvestigation) => {
          this.complaintInvestigationResponse = response;
          this.toastrService.success(this.translationService.getValue('COMPLAINT_INVESTIGATION_CREATED_SUCCESSFULLY'));
          this.complaintInvestigationForm.reset();
          this.complaintInvestigationForm.get('complaintId')?.setValue(this.complaintInvestigationResponse.complaintId);
          this.complaintInvestigationForm.get('actionRequired')?.setValue(false);
        },
      });
    }
  }

  onCancel(): void {
    if (this.data.isViewOnly) {
      this.dialogRef.close();
    } else {
      const complaintId = this.complaintInvestigationForm.get('complaintId')?.value;
      if (complaintId) {
        this.router.navigate(['/complaint/details', complaintId], { queryParams: { isDetailsMode: true } });
      }
    }
  }
}
