import { AfterViewInit, Component, inject, OnInit, ViewChild } from '@angular/core';
import { ComplaintTypeService } from '../complaint-type/complaint-type.service';
import { MatPaginator } from '@angular/material/paginator';
import { MatSort, MatSortModule } from '@angular/material/sort';
import { Subject, debounceTime, distinctUntilChanged, merge, tap } from 'rxjs';
import { UserStore } from '../../user/store/user.store';
import { COMPLAINT_PRIORITY } from '../model/complaint-priority.enum';
import { COMPLAINT_REPORTED_BY } from '../model/complaint-reported-by.enum';
import { COMPLAINT_STATUS } from '../model/complaint-status.enum';
import { ComplaintType } from '../model/complaint-type';
import { ComplaintResource } from '../model/complaint-resource';
import { BaseComponent } from '../../base.component';
import { ComplaintLogService } from './complaint-log.service';
import { HttpResponse } from '@angular/common/http';
import { Complaint } from '../model/complaint';
import { MatTableModule } from '@angular/material/table';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { MatSelectModule } from '@angular/material/select';
import { FormsModule } from '@angular/forms';
import { PageSizeOption } from '@core/utils/global-config';
import { COMPLAINT_LOG_STATUS } from '../model/complaint-log-status.enum';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { TranslateModule } from '@ngx-translate/core';
import { ComplaintPriorityPipe } from '../pipes/complaint-priority.pipe';
import { ComplaintStatusPipe } from '../pipes/complaint-status.pipe';
import { NgClass } from '@angular/common';
import { ComplaintReportedByPipe } from '../pipes/complaint-reported-by.pipe';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { ComplaintLogStatusPipe } from '../pipes/complaint-log-status.pipe';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';

@Component({
  selector: 'app-complaint-log-list',
  imports: [
    MatSortModule,
    MatPaginator,
    MatTableModule,
    MatDatepickerModule,
    MatSelectModule,
    FormsModule,
    PageHelpTextComponent,
    TranslateModule,
    ComplaintPriorityPipe,
    ComplaintStatusPipe,
    NgClass,
    ComplaintReportedByPipe,
    UTCToLocalTime,
    ComplaintLogStatusPipe,
    MatIconModule,
    MatCardModule
  ],
  templateUrl: './complaint-log-list.component.html',
  styleUrl: './complaint-log-list.component.scss'
})
export class ComplaintLogListComponent extends BaseComponent implements OnInit, AfterViewInit {
  displayedColumns: string[] = ['complaintNo', 'title', 'complaintDate', 'assignedToId', 'reportedBy', 'complaintType', 'priority', 'status', 'logStatus'];
  displayedColumnSecondary: string[] = ['search-complaintNo', 'search-title', 'search-complaintDate', 'search-assignedToId', 'search-reportedBy', 'search-complaintType', 'search-priority', 'search-status', 'search-logStatus'];
  footerToDisplayedColumns: string[] = ['footer'];
  complaintTypeList: ComplaintType[] = [];
  complaintList: Complaint[] = [];
  userStore = inject(UserStore);
  complaintTypeService = inject(ComplaintTypeService);
  complaintLogService = inject(ComplaintLogService);
  pageOption = PageSizeOption;

  @ViewChild(MatPaginator) paginator: MatPaginator;
  @ViewChild(MatSort) sort: MatSort;

  statusList = Object.keys(COMPLAINT_STATUS)
    .filter(key => !isNaN(Number(COMPLAINT_STATUS[key as any])))
    .map(key => ({
      name: key,
      id: COMPLAINT_STATUS[key as keyof typeof COMPLAINT_STATUS]
    }));
  priorityList = Object.keys(COMPLAINT_PRIORITY)
    .filter(key => !isNaN(Number(COMPLAINT_PRIORITY[key as any])))
    .map(key => ({
      name: key,
      id: COMPLAINT_PRIORITY[key as keyof typeof COMPLAINT_PRIORITY]
    }));
  reportedByList = Object.keys(COMPLAINT_REPORTED_BY)
    .filter(key => !isNaN(Number(COMPLAINT_REPORTED_BY[key as any])))
    .map(key => ({
      name: key,
      id: COMPLAINT_REPORTED_BY[key as keyof typeof COMPLAINT_REPORTED_BY]
    }));
  logStatus = Object.keys(COMPLAINT_LOG_STATUS)
    .filter(key => !isNaN(Number(COMPLAINT_LOG_STATUS[key as any])))
    .map(key => ({
      id: key,
      name: COMPLAINT_LOG_STATUS[key as keyof typeof COMPLAINT_LOG_STATUS]
    }));


  filterParameter$: Subject<string> = new Subject<string>();
  complaintSearchParameters: ComplaintResource = {
    complaintNo: '',
    title: '',
    complaintFromDate: undefined,
    complaintToDate: undefined,
    reportedBy: '',
    assignedToId: '',
    complaintTypeId: '',
    priority: '',
    status: '',
    skip: 0,
    pageSize: 10,
    orderBy: 'complaintNo asc',
    totalCount: 0,
    logStatus: '',
  };;
  _complaintNoFilter = '';
  _titleFilter = '';
  _complaintFromDateFilter: Date | undefined = undefined;
  _complaintToDateFilter: Date | undefined = undefined;
  _reportedByFilter = '';
  _assignedToFilter = '';
  _complaintTypeFilter = '';
  _priorityFilter = '';
  _statusFilter = '';
  _logStatusFilter = '';

  public get complaintNoFilter(): string {
    return this._complaintNoFilter;
  }

  public set complaintNoFilter(v: string) {
    if (this._complaintNoFilter !== v) {
      this._complaintNoFilter = v;
      const complaintNoFilter = `complaintNo#${v}`;
      this.filterParameter$.next(complaintNoFilter);
    }
  }

  public get titleFilter(): string {
    return this._titleFilter;
  }

  public set titleFilter(v: string) {
    if (this._titleFilter !== v) {
      this._titleFilter = v;
      const titleFilter = `title#${v}`;
      this.filterParameter$.next(titleFilter);
    }
  }

  public get complaintFromDateFilter(): Date | undefined {
    return this._complaintFromDateFilter;
  }

  public set complaintFromDateFilter(v: Date | undefined) {
    if (this._complaintFromDateFilter !== v) {
      this._complaintFromDateFilter = v;
      this.complaintSearchParameters.complaintFromDate = v;
      this.filterParameter$.next(`complaintFromDate#${v}`);
    }
  }

  public get complaintToDateFilter(): Date | undefined {
    return this._complaintToDateFilter;
  }

  public set complaintToDateFilter(v: Date | undefined) {
    if (this._complaintToDateFilter !== v) {
      this._complaintToDateFilter = v;
      this.complaintSearchParameters.complaintToDate = v;
      this.filterParameter$.next(`complaintToDate#${v}`);
    }
  }

  public get reportedByFilter(): string {
    return this._reportedByFilter;
  }

  public set reportedByFilter(v: string) {
    if (this._reportedByFilter !== v) {
      this._reportedByFilter = v;
      this.complaintSearchParameters.reportedBy = v;
      this.filterParameter$.next(`reportedBy#${v}`);
    }
  }

  public get assignedToFilter(): string {
    return this._assignedToFilter;
  }

  public set assignedToFilter(v: string) {
    if (this._assignedToFilter !== v) {
      this._assignedToFilter = v;
      this.complaintSearchParameters.assignedToId = v;
      this.filterParameter$.next(`assignedToId#${v}`);
    }
  }

  public get complaintTypeFilter(): string {
    return this._complaintTypeFilter;
  }

  public set complaintTypeFilter(v: string) {
    if (this._complaintTypeFilter !== v) {
      this._complaintTypeFilter = v;
      this.complaintSearchParameters.complaintTypeId = v;
      this.filterParameter$.next(`complaintTypeId#${v}`);
    }
  }

  public get priorityFilter(): string {
    return this._priorityFilter;
  }

  public set priorityFilter(v: string) {
    if (this._priorityFilter !== v) {
      this._priorityFilter = v;
      this.complaintSearchParameters.priority = v;
      this.filterParameter$.next(`priority#${v}`);
    }
  }

  public get statusFilter(): string {
    return this._statusFilter;
  }

  public set statusFilter(v: string) {
    if (this._statusFilter !== v) {
      this._statusFilter = v;
      this.complaintSearchParameters.status = v;
      this.filterParameter$.next(`status#${v}`);
    }
  }

  public get logStatusFilter(): string {
    return this._logStatusFilter;
  }

  public set logStatusFilter(v: string) {
    if (this._logStatusFilter !== v) {
      this._logStatusFilter = v;
      this.complaintSearchParameters.logStatus = v;
      this.filterParameter$.next(`logStatus#${v}`);
    }
  }

  clearOrderDates() {
    this._complaintToDateFilter = undefined;
    this._complaintFromDateFilter = undefined;
  }

  ngOnInit(): void {
    this.getAllCompamintTypes();
    this.loadByQuery(this.complaintSearchParameters);

    this.sub$.sink = this.filterParameter$
      .pipe(
        debounceTime(1000),
        distinctUntilChanged())
      .subscribe((c: string) => {
        this.complaintSearchParameters.skip = 0;
        this.paginator.pageIndex = 0;
        const filterArray: Array<string> = c.split('#');
        if (filterArray[0] == 'complaintNo') {
          this.complaintSearchParameters.complaintNo = filterArray[1];
        } else if (filterArray[0] === 'title') {
          this.complaintSearchParameters.title = filterArray[1];
        } else if (filterArray[0] === 'complaintFromDate') {
          if (filterArray[1] !== 'null') {
            this.complaintSearchParameters.complaintFromDate = new Date(filterArray[1]);
            this.complaintSearchParameters.complaintToDate = this.complaintToDateFilter;
          } else {
            this.complaintSearchParameters.complaintFromDate = undefined;
            this.complaintSearchParameters.complaintToDate = undefined;
          }
        } else if (filterArray[0] === 'complaintToDate') {
          if (filterArray[1] !== 'null') {
            this.complaintSearchParameters.complaintToDate = new Date(filterArray[1]);
            this.complaintSearchParameters.complaintFromDate = this.complaintFromDateFilter;
          } else {
            this.complaintSearchParameters.complaintFromDate = undefined;
            this.complaintSearchParameters.complaintToDate = undefined;
          }
        } else if (filterArray[0] === 'reportedBy') {
          this.complaintSearchParameters.reportedBy = filterArray[1];
        } else if (filterArray[0] === 'assignedToId') {
          this.complaintSearchParameters.assignedToId = filterArray[1];
        } else if (filterArray[0] === 'complaintTypeId') {
          this.complaintSearchParameters.complaintTypeId = filterArray[1];
        } else if (filterArray[0] === 'priority') {
          this.complaintSearchParameters.priority = filterArray[1];
        } else if (filterArray[0] === 'status') {
          this.complaintSearchParameters.status = filterArray[1];
        } else if (filterArray[0] === 'logStatus') {
          this.complaintSearchParameters.logStatus = filterArray[1];
        }
        this.loadByQuery(this.complaintSearchParameters);
      });
  }

  getAllCompamintTypes() {
    this.sub$.sink = this.complaintTypeService.getComplaintTypes().subscribe({
      next: (complaintTypes) => {
        const types = complaintTypes as ComplaintType[];
        if (types) {
          this.complaintTypeList = [...types];
        }
      },
    })
  }

  ngAfterViewInit() {
    this.sub$.sink = this.sort.sortChange.subscribe(() => {
      this.paginator.pageIndex = 0
    });
    this.sub$.sink = merge(this.sort.sortChange, this.paginator.page)
      .pipe(
        tap(() => {
          this.complaintSearchParameters.skip = this.paginator.pageIndex * this.paginator.pageSize;
          this.complaintSearchParameters.pageSize = this.paginator.pageSize;
          this.complaintSearchParameters.orderBy = this.sort.active ? `${this.sort.active} ${this.sort.direction}` : 'complaintNo asc';
          this.loadByQuery(this.complaintSearchParameters);
        }),
      ).subscribe();
  }

  loadByQuery(filter: ComplaintResource) {
    return this.complaintLogService.getComplaints(filter).subscribe({
      next: (httpResponse: HttpResponse<Complaint[]>) => {
        if (httpResponse && httpResponse.body) {
          this.complaintList = [...httpResponse.body as Complaint[]];
        }
        const paginationHeader = httpResponse.headers.get('x-pagination');
        if (paginationHeader) {
          const pagination = JSON.parse(paginationHeader);
          this.complaintSearchParameters = { ...filter, totalCount: pagination.totalCount };
        }
      },
    })
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.complaintList.indexOf(row);
  }
}
