import { Component, Inject, inject, OnInit, Optional } from '@angular/core';
import { FormBuilder, FormGroup, ReactiveFormsModule, Validators } from '@angular/forms';
import { ActivatedRoute, RouterLink } from '@angular/router';
import { TextEditorComponent } from '@shared/text-editor/text-editor.component';
import { ComplaintRootCauseListComponent } from "../complaint-root-cause-list/complaint-root-cause-list.component";
import { BaseComponent } from '../../../base.component';
import { MatCardModule } from '@angular/material/card';
import { ComplaintRootCause } from '../../model/complaint-root-cause';
import { MatButtonModule } from '@angular/material/button';
import { MAT_DIALOG_DATA, MatDialogRef } from '@angular/material/dialog';
import { NgClass } from '@angular/common';
import { ComplaintRootCauseService } from '../complaint-root-cause.service';

import { ToastrService } from '@core/services/toastr-service';
import { TranslateModule } from '@ngx-translate/core';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { MatIconModule } from '@angular/material/icon';
import { RichTextRequired } from '@shared/validators/rich-text.validator';

@Component({
  selector: 'app-manage-complaint-root-cause',
  imports: [
    MatIconModule,
    ReactiveFormsModule,
    TextEditorComponent,
    RouterLink,
    ComplaintRootCauseListComponent,
    MatCardModule,
    MatButtonModule,
    NgClass,
    TranslateModule,
    HasClaimDirective
  ],
  templateUrl: './manage-complaint-root-cause.component.html',
  styleUrl: './manage-complaint-root-cause.component.scss'
})
export class ManageComplaintRootCauseComponent extends BaseComponent implements OnInit {
  complaintRootCauseForm: FormGroup;
  formBuilder = inject(FormBuilder);
  complaintRootCauses: ComplaintRootCause = {} as ComplaintRootCause;
  complaintRootCauseService = inject(ComplaintRootCauseService);
  toastrService = inject(ToastrService);
  route = inject(ActivatedRoute);

  constructor(
    @Optional() private dialogRef: MatDialogRef<ManageComplaintRootCauseComponent>,
    @Optional() @Inject(MAT_DIALOG_DATA) public data: { complaintRootCause: ComplaintRootCause, isViewOnly: boolean },
  ) {
    super();
  }

  ngOnInit(): void {
    this.creatComplaintRootCauseForm();

    this.route.parent?.paramMap.subscribe(params => {
      const complaintId = params.get('id') || '';
      if (complaintId) {
        this.complaintRootCauseForm.get('complaintId')?.setValue(complaintId);
      }
    });

    if (this.data !== null) {
      this.complaintRootCauseForm.patchValue(this.data.complaintRootCause);
    } else {
      this.data = {
        complaintRootCause: {} as ComplaintRootCause,
        isViewOnly: false
      }
    }
  }

  creatComplaintRootCauseForm(): void {
    this.complaintRootCauseForm = this.formBuilder.group({
      id: [''],
      complaintId: [''],
      methodUsed: ['', [Validators.required]],
      causeDescription: ['', [RichTextRequired]],
    });
  }

  onSubmit(): void {
    if (!this.complaintRootCauseForm.valid) {
      this.complaintRootCauseForm.markAllAsTouched();
      return;
    }

    if (this.complaintRootCauseForm.invalid) return;

    const complaintRootCauseData = this.complaintRootCauseForm.getRawValue();

    this.sub$.sink = this.complaintRootCauseService.createComplaintRootCause(complaintRootCauseData).subscribe({
      next: (response: ComplaintRootCause) => {
        this.complaintRootCauses = response;
        this.toastrService.success(this.translationService.getValue('COMPLAINT_ROOT_CAUSE_CREATED_SUCCESSFULLY'));
        this.complaintRootCauseForm.reset();
        this.complaintRootCauseForm.get('complaintId')?.setValue(response.complaintId);
      }
    });
  }

  onCancel(): void {
    this.dialogRef.close();
  }
}

