import { Component, inject, OnInit } from '@angular/core';
import { MatButtonModule } from '@angular/material/button';
import { MatDialog } from '@angular/material/dialog';
import { MatTableModule } from '@angular/material/table';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';

import { ToastrService } from '@core/services/toastr-service';
import { ComplaintType } from '../../model/complaint-type';
import { ComplaintTypeService } from '../complaint-type.service';
import { ManageComplaintTypeComponent } from '../manage-complaint-type/manage-complaint-type.component';
import { BaseComponent } from '../../../base.component';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { TranslateModule } from '@ngx-translate/core';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';
import { NgClass } from '@angular/common';

@Component({
  selector: 'app-complaint-type-list',
  imports: [
    MatButtonModule,
    MatIconModule,
    MatTableModule,
    PageHelpTextComponent,
    HasClaimDirective,
    PageHelpTextComponent,
    TranslateModule,
    MatCardModule,
    NgClass
  ],
  templateUrl: './complaint-type-list.component.html',
  styleUrl: './complaint-type-list.component.scss'
})
export class ComplaintTypeListComponent extends BaseComponent implements OnInit {
  complaintTypes: ComplaintType[] = [];
  dialog = inject(MatDialog);
  displayedColumns: string[] = ['action', 'name', 'description'];
  commandDialogService = inject(CommonDialogService);
  toastrService = inject(ToastrService);
  complaintTypeService = inject(ComplaintTypeService);

  ngOnInit(): void {
    this.getAllComplaintTypes();
  }

  getAllComplaintTypes() {
    this.sub$.sink = this.complaintTypeService.getComplaintTypes().subscribe({
      next: (data: ComplaintType[]) => {
        this.complaintTypes = [...data];
      },
    });
  }

  addComplaintType() {
    const dialogRef = this.dialog.open(ManageComplaintTypeComponent, {
      width: '500px',
      data: null
    });
    this.sub$.sink = dialogRef.afterClosed().subscribe((result: ComplaintType) => {
      if (result) {
        this.complaintTypes = [result, ...this.complaintTypes];
      }
    });
  }

  editComplaintType(type: ComplaintType) {
    const dialogRef = this.dialog.open(ManageComplaintTypeComponent, {
      width: '500px',
      data: type
    });
    this.sub$.sink = dialogRef.afterClosed().subscribe((result: ComplaintType) => {
      if (result) {
        this.complaintTypes = this.complaintTypes.map(t => t.id === result.id ? { ...result } : t);
      }
    });
  }

  deleteComplaintType(typeId: string) {
    this.commandDialogService.deleteConfirmtionDialog(this.translationService.getValue("ARE_YOU_SURE_YOU_WANT_TO_DELETE_THIS_TYPE")).subscribe({
      next: (result: boolean) => {
        if (result) {
          this.sub$.sink = this.complaintTypeService.deleteComplaintType(typeId).subscribe({
            next: () => {
              this.toastrService.success(this.translationService.getValue("COMPLAINT_TYPE_DELETED_SUCCESSFULLY"));
              this.complaintTypes = this.complaintTypes.filter(type => type.id !== typeId);
            },
          });
        };
      }
    });
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.complaintTypes.indexOf(row);
  }
}
