import { Component, inject, Inject } from '@angular/core';
import { FormGroup, FormBuilder, Validators, ReactiveFormsModule } from '@angular/forms';
import { MatDialogRef, MAT_DIALOG_DATA, MatDialogModule } from '@angular/material/dialog';
import { ComplaintTypeService } from '../complaint-type.service';

import { ToastrService } from '@core/services/toastr-service';
import { TranslationService } from '@core/services/translation.service';
import { ComplaintType } from '../../model/complaint-type';
import { MatButtonModule } from '@angular/material/button';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { TranslateModule } from '@ngx-translate/core';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';

@Component({
  selector: 'app-manage-complaint-type',
  standalone: true,
  imports: [
    MatIconModule,
    ReactiveFormsModule,
    MatDialogModule,
    MatButtonModule,
    PageHelpTextComponent,
    TranslateModule,
    MatCardModule
  ],
  templateUrl: './manage-complaint-type.component.html',
  styleUrl: './manage-complaint-type.component.scss'
})
export class ManageComplaintTypeComponent {
  complaintTypeForm: FormGroup;
  isEdit = false;
  fb = inject(FormBuilder);
  complaintTypeService = inject(ComplaintTypeService);
  toastrService = inject(ToastrService);
  translationService = inject(TranslationService);

  constructor(
    private dialogRef: MatDialogRef<ManageComplaintTypeComponent>,
    @Inject(MAT_DIALOG_DATA) public complaintDetail: ComplaintType
  ) { }

  ngOnInit(): void {
    this.createCategoryForm();
    if (this.complaintDetail) {
      this.isEdit = true;
      this.complaintTypeForm.patchValue(this.complaintDetail);
    }
  }

  createCategoryForm() {
    this.complaintTypeForm = this.fb.group({
      id: [''],
      name: ['', [Validators.required]],
      description: [''],
    });
  }

  onSubmit() {
    if (!this.complaintTypeForm.valid) {
      this.complaintTypeForm.markAllAsTouched();
      return;
    }

    const category = this.complaintTypeForm.getRawValue();

    if (this.isEdit) {
      this.complaintTypeService.updateComplaintType(category).subscribe({
        next: (result) => {
          this.toastrService.success(this.translationService.getValue('COMPLAINT_TYPE_UPDATED_SUCCESSFULLY'));
          this.dialogRef.close(result);
        },
      });
    } else {
      this.complaintTypeService.addComplaintType(category).subscribe({
        next: (result) => {
          this.toastrService.success(this.translationService.getValue('COMPLAINT_TYPE_ADDED_SUCCESSFULLY'));
          this.dialogRef.close(result);
        },
      });
    }
  }

  onCancel() {
    this.dialogRef.close();
  }
}
