import { AfterViewInit, Component, inject, OnInit, ViewChild } from '@angular/core';
import { MatButtonModule } from '@angular/material/button';
import { MatPaginator } from '@angular/material/paginator';
import { MatSelectModule } from '@angular/material/select';
import { MatSort, MatSortModule } from '@angular/material/sort';
import { MatTableModule } from '@angular/material/table';
import { RouterLink } from '@angular/router';
import { ComplaintStore } from '../complaint.store';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { UserStore } from '../../../user/store/user.store';
import { BaseComponent } from '../../../base.component';
import { ComplaintTypeService } from '../../complaint-type/complaint-type.service';
import { ComplaintType } from '../../model/complaint-type';
import { COMPLAINT_STATUS } from '../../model/complaint-status.enum';
import { COMPLAINT_PRIORITY } from '../../model/complaint-priority.enum';
import { COMPLAINT_REPORTED_BY } from '../../model/complaint-reported-by.enum';
import { MatMenuModule } from '@angular/material/menu';
import { MatIconModule } from '@angular/material/icon';
import { debounceTime, distinctUntilChanged, merge, Subject, tap } from 'rxjs';
import { FormsModule } from '@angular/forms';

import { ToastrService } from '@core/services/toastr-service';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { TranslateModule } from '@ngx-translate/core';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { ComplaintPriorityPipe } from '../../pipes/complaint-priority.pipe';
import { ComplaintStatusPipe } from '../../pipes/complaint-status.pipe';
import { NgClass } from '@angular/common';
import { ComplaintReportedByPipe } from '../../pipes/complaint-reported-by.pipe';
import { MatCardModule } from '@angular/material/card';

@Component({
  selector: 'app-complaint-list',
  imports: [
    MatButtonModule,
    MatSelectModule,
    RouterLink,
    MatTableModule,
    MatSortModule,
    MatPaginator,
    MatDatepickerModule,
    MatMenuModule,
    MatIconModule,
    FormsModule,
    PageHelpTextComponent,
    TranslateModule,
    HasClaimDirective,
    UTCToLocalTime,
    ComplaintPriorityPipe,
    ComplaintStatusPipe,
    NgClass,
    ComplaintReportedByPipe,
    MatCardModule
  ],
  templateUrl: './complaint-list.component.html',
  styleUrl: './complaint-list.component.scss'
})
export class ComplaintListComponent extends BaseComponent implements OnInit, AfterViewInit {
  displayedColumns: string[] = ['actions', 'complaintNo', 'title', 'complaintDate', 'assignedToId', 'reportedBy', 'complaintType', 'priority', 'status',];
  displayedColumnSecondary: string[] = ['search-actions', 'search-complaintNo', 'search-title', 'search-complaintDate', 'search-assignedToId', 'search-reportedBy', 'search-complaintType', 'search-priority', 'search-status',];
  footerToDisplayedColumns: string[] = ['footer'];
  complaintTypeList: ComplaintType[] = [];
  toastrService = inject(ToastrService);
  complaintStore = inject(ComplaintStore);
  userStore = inject(UserStore);
  complaintTypeService = inject(ComplaintTypeService);
  commandDialogService = inject(CommonDialogService);

  @ViewChild(MatPaginator) paginator: MatPaginator;
  @ViewChild(MatSort) sort: MatSort;

  statusList = Object.keys(COMPLAINT_STATUS)
    .filter(key => !isNaN(Number(COMPLAINT_STATUS[key as any])))
    .map(key => ({
      name: key,
      id: COMPLAINT_STATUS[key as keyof typeof COMPLAINT_STATUS]
    }));
  priorityList = Object.keys(COMPLAINT_PRIORITY)
    .filter(key => !isNaN(Number(COMPLAINT_PRIORITY[key as any])))
    .map(key => ({
      name: key,
      id: COMPLAINT_PRIORITY[key as keyof typeof COMPLAINT_PRIORITY]
    }));
  reportedByList = Object.keys(COMPLAINT_REPORTED_BY)
    .filter(key => !isNaN(Number(COMPLAINT_REPORTED_BY[key as any])))
    .map(key => ({
      name: key,
      id: COMPLAINT_REPORTED_BY[key as keyof typeof COMPLAINT_REPORTED_BY]
    }));

  filterParameter$: Subject<string> = new Subject<string>();
  complaintSearchParameters = this.complaintStore.filterParameters();
  _complaintNoFilter = this.complaintSearchParameters.complaintNo;
  _titleFilter = this.complaintSearchParameters.title;
  _complaintFromDateFilter = this.complaintSearchParameters.complaintFromDate;
  _complaintToDateFilter = this.complaintSearchParameters.complaintToDate;
  _reportedByFilter = this.complaintSearchParameters.reportedBy;
  _assignedToFilter = this.complaintSearchParameters.assignedToId;
  _complaintTypeFilter = this.complaintSearchParameters.complaintTypeId;
  _priorityFilter = this.complaintSearchParameters.priority;
  _statusFilter = this.complaintSearchParameters.status;

  public get complaintNoFilter(): string {
    return this._complaintNoFilter;
  }

  public set complaintNoFilter(v: string) {
    if (this._complaintNoFilter !== v) {
      this._complaintNoFilter = v;
      const complaintNoFilter = `complaintNo#${v}`;
      this.filterParameter$.next(complaintNoFilter);
    }
  }

  public get titleFilter(): string {
    return this._titleFilter;
  }

  public set titleFilter(v: string) {
    if (this._titleFilter !== v) {
      this._titleFilter = v;
      const titleFilter = `title#${v}`;
      this.filterParameter$.next(titleFilter);
    }
  }

  public get complaintFromDateFilter(): Date | undefined {
    return this._complaintFromDateFilter;
  }

  public set complaintFromDateFilter(v: Date) {
    if (this._complaintFromDateFilter !== v) {
      this._complaintFromDateFilter = v;
      this.complaintSearchParameters.complaintFromDate = v;
      this.filterParameter$.next(`complaintFromDate#${v}`);
    }
  }

  public get complaintToDateFilter(): Date | undefined {
    return this._complaintToDateFilter;
  }

  public set complaintToDateFilter(v: Date) {
    if (this._complaintToDateFilter !== v) {
      this._complaintToDateFilter = v;
      this.complaintSearchParameters.complaintToDate = v;
      this.filterParameter$.next(`complaintToDate#${v}`);
    }
  }

  public get reportedByFilter(): string {
    return this._reportedByFilter;
  }

  public set reportedByFilter(v: string) {
    if (this._reportedByFilter !== v) {
      this._reportedByFilter = v;
      this.complaintSearchParameters.reportedBy = v;
      this.filterParameter$.next(`reportedBy#${v}`);
    }
  }

  public get assignedToFilter(): string {
    return this._assignedToFilter;
  }

  public set assignedToFilter(v: string) {
    if (this._assignedToFilter !== v) {
      this._assignedToFilter = v;
      this.complaintSearchParameters.assignedToId = v;
      this.filterParameter$.next(`assignedToId#${v}`);
    }
  }

  public get complaintTypeFilter(): string {
    return this._complaintTypeFilter;
  }

  public set complaintTypeFilter(v: string) {
    if (this._complaintTypeFilter !== v) {
      this._complaintTypeFilter = v;
      this.complaintSearchParameters.complaintTypeId = v;
      this.filterParameter$.next(`complaintTypeId#${v}`);
    }
  }

  public get priorityFilter(): string {
    return this._priorityFilter;
  }

  public set priorityFilter(v: string) {
    if (this._priorityFilter !== v) {
      this._priorityFilter = v;
      this.complaintSearchParameters.priority = v;
      this.filterParameter$.next(`priority#${v}`);
    }
  }

  public get statusFilter(): string {
    return this._statusFilter;
  }

  public set statusFilter(v: string) {
    if (this._statusFilter !== v) {
      this._statusFilter = v;
      this.complaintSearchParameters.status = v;
      this.filterParameter$.next(`status#${v}`);
    }
  }

  clearOrderDates() {
    this._complaintFromDateFilter = undefined;
    this._complaintToDateFilter = undefined;
  }

  ngOnInit(): void {
    this.getAllCompamintTypes();

    this.sub$.sink = this.filterParameter$
      .pipe(debounceTime(1000), distinctUntilChanged())
      .subscribe((c: string) => {
        this.complaintSearchParameters.skip = 0;
        this.paginator.pageIndex = 0;
        const filterArray: Array<string> = c.split('#');
        if (filterArray[0] == 'complaintNo') {
          this.complaintSearchParameters.complaintNo = filterArray[1];
        } else if (filterArray[0] === 'title') {
          this.complaintSearchParameters.title = filterArray[1];
        } else if (filterArray[0] === 'complaintFromDate') {
          if (filterArray[1] !== 'null') {
            this.complaintSearchParameters.complaintFromDate = new Date(filterArray[1]);
            this.complaintSearchParameters.complaintToDate = this.complaintToDateFilter;
          } else {
            this.complaintSearchParameters.complaintFromDate = undefined;
            this.complaintSearchParameters.complaintToDate = undefined;
          }
        } else if (filterArray[0] === 'complaintToDate') {
          if (filterArray[1] !== 'null') {
            this.complaintSearchParameters.complaintToDate = new Date(filterArray[1]);
            this.complaintSearchParameters.complaintFromDate = this.complaintFromDateFilter;
          } else {
            this.complaintSearchParameters.complaintFromDate = undefined;
            this.complaintSearchParameters.complaintToDate = undefined;
          }
        } else if (filterArray[0] === 'reportedBy') {
          this.complaintSearchParameters.reportedBy = filterArray[1];
        } else if (filterArray[0] === 'assignedToId') {
          this.complaintSearchParameters.assignedToId = filterArray[1];
        } else if (filterArray[0] === 'complaintTypeId') {
          this.complaintSearchParameters.complaintTypeId = filterArray[1];
        } else if (filterArray[0] === 'priority') {
          this.complaintSearchParameters.priority = filterArray[1];
        } else if (filterArray[0] === 'status') {
          this.complaintSearchParameters.status = filterArray[1];
        }
        this.refresh();
      });
  }

  getAllCompamintTypes() {
    this.sub$.sink = this.complaintTypeService.getComplaintTypes().subscribe({
      next: (complaintTypes) => {
        const types = complaintTypes as ComplaintType[];
        if (types) {
          this.complaintTypeList = [...types];
        }
      },
    })
  }

  deleteComplaint(id: string) {
    this.commandDialogService.deleteConfirmtionDialog(this.translationService.getValue("ARE_YOU_SURE_YOU_WANT_TO_DELETE_THIS_COMPLAINT")).subscribe({
      next: (result: boolean) => {
        if (result) {
          this.complaintStore.deleteComplaintById(id);
        };
      }
    });
  }

  refresh() {
    this.complaintStore.loadByQuery(this.complaintSearchParameters);
  }

  ngAfterViewInit() {
    this.sub$.sink = this.sort.sortChange.subscribe(() => {
      this.paginator.pageIndex = 0
    });
    this.sub$.sink = merge(this.sort.sortChange, this.paginator.page)
      .pipe(
        tap(() => {
          this.complaintSearchParameters.skip = this.paginator.pageIndex * this.paginator.pageSize;
          this.complaintSearchParameters.pageSize = this.paginator.pageSize;
          this.complaintSearchParameters.orderBy = this.sort.active ? `${this.sort.active} ${this.sort.direction}` : 'complaintNo asc';
          this.refresh();
        }),
      ).subscribe();
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.complaintStore.complaintList().indexOf(row);
  }
}
