import { Injectable, inject } from '@angular/core';
import { HttpClient, HttpParams, HttpResponse } from '@angular/common/http';
import { Observable } from 'rxjs';
import { catchError } from 'rxjs/operators';
import { CommonError } from '@core/error-handler/common-error';
import { CommonHttpErrorService } from '@core/error-handler/common-http-error.service';
import { Complaint } from '../model/complaint';
import { ComplaintResource } from '../model/complaint-resource';

@Injectable({
  providedIn: 'root'
})
export class ComplaintService {
  private httpClient = inject(HttpClient);
  private commonHttpErrorService = inject(CommonHttpErrorService);

  constructor() { }

  getComplaints(complaintResource: ComplaintResource): Observable<HttpResponse<Complaint[]>> {
    const url = 'Complaint';
    const params = new HttpParams()
      .set('orderBy', complaintResource.orderBy)
      .set('pageSize', complaintResource.pageSize.toString())
      .set('skip', complaintResource.skip.toString())
      .set('totalCount', complaintResource.totalCount.toString())
      .set('title', complaintResource.title || '')
      .set('complaintNo', complaintResource.complaintNo || '')
      .set('assignedToId', complaintResource.assignedToId || '')
      .set('reportedBy', complaintResource.reportedBy || '')
      .set('complaintTypeId', complaintResource.complaintTypeId || '')
      .set('complaintFromDate', complaintResource.complaintFromDate ? complaintResource.complaintFromDate.toISOString() : '')
      .set('complaintToDate', complaintResource.complaintToDate ? complaintResource.complaintToDate.toISOString() : '')
      .set('priority', complaintResource.priority || '')
      .set('status', complaintResource.status?.toString() || '')
    return this.httpClient
      .get<Complaint[]>(url, {
        params: params,
        observe: 'response'
      });

  }

  createComplaint(complaint: Complaint): Observable<Complaint | CommonError> {
    const url = 'Complaint';
    return this.httpClient
      .post<Complaint>(url, complaint)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  updateComplaint(complaint: Complaint): Observable<Complaint | CommonError> {
    const url = `Complaint/${complaint.id}`;
    return this.httpClient
      .put<Complaint>(url, complaint)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  deleteComplaint(complaintId: string): Observable<void | CommonError> {
    const url = `Complaint/${complaintId}`;
    return this.httpClient
      .delete<void>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getComplaintById(complaintId: string): Observable<Complaint | CommonError> {
    const url = `Complaint/${complaintId}`;
    return this.httpClient
      .get<Complaint>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }
}
