import { inject } from "@angular/core";
import { toObservable } from "@angular/core/rxjs-interop";
import { TranslationService } from "@core/services/translation.service";
import { tapResponse } from "@ngrx/operators";
import { signalStore, withState, withMethods, patchState, withHooks } from "@ngrx/signals";
import { rxMethod } from "@ngrx/signals/rxjs-interop";
import { ToastrService } from '@core/services/toastr-service';
import { pipe, tap, switchMap, distinctUntilChanged } from "rxjs";
import { Complaint } from "../model/complaint";
import { ComplaintResource } from "../model/complaint-resource";
import { ComplaintService } from "./complaint.service";
import { HttpResponse } from "@angular/common/http";

type ComplaintState = {
  complaintList: Complaint[];
  complaint: Complaint;
  loadList: boolean;
  isAddUpdate: boolean;
  filterParameters: ComplaintResource;
};

export const initialComplaintState: ComplaintState = {
  complaintList: [],
  complaint: {} as Complaint,
  loadList: true,
  isAddUpdate: false,
  filterParameters: {
    orderBy: 'complaintNo asc',
    pageSize: 10,
    skip: 0,
    totalCount: 0,
    title: '',
    complaintNo: '',
    assignedToId: '',
    reportedBy: '',
    complaintTypeId: '',
    complaintFromDate: undefined,
    complaintToDate: undefined,
    priority: '',
    status: '',
  }
};

export const ComplaintStore = signalStore(
  { providedIn: 'root' },
  withState(initialComplaintState),
  withMethods(
    (
      store,
      complaintService = inject(ComplaintService),
      toastrService = inject(ToastrService),
      translationService = inject(TranslationService),
    ) => ({
      loadByQuery: rxMethod<ComplaintResource>(
        pipe(
          tap(() => patchState(store, { complaintList: [] })),
          switchMap((filter) => {
            return complaintService.getComplaints(filter).pipe(
              tapResponse({
                next: (httpResponse: HttpResponse<Complaint[]>) => {
                  if (httpResponse && httpResponse.body) {
                    patchState(store, { complaintList: [...httpResponse.body], loadList: false })
                  }
                  if (httpResponse.headers.get('x-pagination')) {
                    const pagination: ComplaintResource = JSON.parse(httpResponse.headers.get('x-pagination') ?? '');
                    patchState(store, { filterParameters: { ...filter, totalCount: pagination.totalCount } })
                  }
                },
                error: (error: any) => {
                  toastrService.error(error.error);
                }
              })
            );
          })
        )
      ),
      addComplaint: rxMethod<Complaint>(
        pipe(
          distinctUntilChanged(),
          tap(() => patchState(store, { loadList: false })),
          switchMap((complaint: Complaint) =>
            complaintService.createComplaint(complaint).pipe(
              tapResponse({
                next: () => {
                  toastrService.success(
                    translationService.getValue('COMPLAINT_CREATED_SUCCESSFULLY')
                  );
                  patchState(store, { isAddUpdate: true, loadList: true });
                },
                error: (err: any) => {
                  console.error(err);
                },
              })
            )
          )
        )
      ),
      updateComplaint: rxMethod<Complaint>(
        pipe(
          distinctUntilChanged(),
          tap(() => patchState(store, { loadList: false })),
          switchMap((complaint: Complaint) =>
            complaintService.updateComplaint(complaint).pipe(
              tapResponse({
                next: (response) => {
                  const updatedComplaint: Complaint = response as Complaint;
                  toastrService.success(
                    translationService.getValue('COMPLAINT_UPDATED_SUCCESSFULLY')
                  );
                  patchState(store, {
                    complaintList: store.complaintList().map((cr) => {
                      return updatedComplaint.id === cr.id ? updatedComplaint : cr;
                    }),
                    isAddUpdate: true,
                  });
                },
                error: (err: any) => {
                  console.error(err);
                },
              })
            )
          )
        )
      ),
      deleteComplaintById: rxMethod<string>(
        pipe(
          distinctUntilChanged(),
          tap(() => patchState(store, { loadList: false })),
          switchMap((complaintId: string) =>
            complaintService.deleteComplaint(complaintId).pipe(
              tapResponse({
                next: () => {
                  toastrService.success(
                    translationService.getValue('COMPLAINT_DELETED_SUCCESSFULLY')
                  );
                  patchState(store, {
                    complaintList: store.complaintList().filter((w) => w.id !== complaintId),
                    filterParameters: {
                      ...store.filterParameters(),
                      totalCount: store.filterParameters().totalCount - 1,
                    },
                  });
                },
                error: (err: any) => {
                  console.error(err);
                },
              })
            )
          )
        )
      ),
      resetflag() {
        patchState(store, { isAddUpdate: false })
      },
    })
  ),
  withHooks({
    onInit(store) {
      toObservable(store.loadList).subscribe((flag) => {
        if (flag) {
          store.loadByQuery(store.filterParameters());
        }
      });
    },
  })
);
