import { NgClass } from '@angular/common';
import { Component, inject, OnInit } from '@angular/core';
import { FormBuilder, FormGroup, FormsModule, ReactiveFormsModule, Validators, AbstractControl, ValidationErrors } from '@angular/forms';
import { MatCardModule } from '@angular/material/card';
import { MatSelectModule } from '@angular/material/select';
import { TextEditorComponent } from '@shared/text-editor/text-editor.component';
import { BaseComponent } from '../../../base.component';
import { COMPLAINT_PRIORITY } from '../../model/complaint-priority.enum';
import { COMPLAINT_REPORTED_BY } from '../../model/complaint-reported-by.enum';
import { COMPLAINT_STATUS } from '../../model/complaint-status.enum';
import { ComplaintType } from '../../model/complaint-type';
import { ComplaintTypeService } from '../../complaint-type/complaint-type.service';
import { MatButtonModule } from '@angular/material/button';
import { UserStore } from '../../../user/store/user.store';
import { ActivatedRoute, Router, RouterLink } from '@angular/router';
import { ComplaintStore } from '../complaint.store';
import { toObservable } from '@angular/core/rxjs-interop';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { Complaint } from '../../model/complaint';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { TranslateModule } from '@ngx-translate/core';
import { MatIconModule } from '@angular/material/icon';
import { MatDialog } from '@angular/material/dialog';
import { ManageComplaintTypeComponent } from '../../complaint-type/manage-complaint-type/manage-complaint-type.component';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { RichTextRequired } from '@shared/validators/rich-text.validator';

@Component({
  selector: 'app-manage-complaint',
  imports: [
    MatSelectModule,
    NgClass,
    MatCardModule,
    ReactiveFormsModule,
    TextEditorComponent,
    MatButtonModule,
    MatDatepickerModule,
    RouterLink,
    PageHelpTextComponent,
    TranslateModule,
    MatIconModule,
    HasClaimDirective
  ],
  templateUrl: './manage-complaint.component.html',
  styleUrl: './manage-complaint.component.scss'
})
export class ManageComplaintComponent extends BaseComponent implements OnInit {
  complaintForm: FormGroup;
  fb = inject(FormBuilder);
  isEdit: boolean = false;
  isDetailsMode: boolean = true;
  complaintTypeService = inject(ComplaintTypeService);
  complaintTypeList: ComplaintType[] = [];
  complaintStore = inject(ComplaintStore);
  userStore = inject(UserStore);
  route = inject(ActivatedRoute);
  router = inject(Router);
  minDueDate: Date = new Date();
  dialog = inject(MatDialog);

  statusList = Object.keys(COMPLAINT_STATUS)
    .filter(key => !isNaN(Number(COMPLAINT_STATUS[key as any])))
    .map(key => ({
      name: key,
      id: COMPLAINT_STATUS[key as keyof typeof COMPLAINT_STATUS]
    }));

  priorityList = Object.keys(COMPLAINT_PRIORITY)
    .filter(key => !isNaN(Number(COMPLAINT_PRIORITY[key as any])))
    .map(key => ({
      name: key,
      id: COMPLAINT_PRIORITY[key as keyof typeof COMPLAINT_PRIORITY]
    }));

  reportedByList = Object.keys(COMPLAINT_REPORTED_BY)
    .filter(key => !isNaN(Number(COMPLAINT_REPORTED_BY[key as any])))
    .map(key => ({
      name: key,
      id: COMPLAINT_REPORTED_BY[key as keyof typeof COMPLAINT_REPORTED_BY]
    }));

  constructor() {
    super();
    this.subscribeIsAddorUpdate();
  }

  ngOnInit(): void {
    this.createComplaintForm();
    this.getAllCompamintTypes();

    this.sub$.sink = this.route.data.subscribe((params) => {
      const complaint: Complaint = params['complaintDetails']
      if (complaint) {
        this.isEdit = true;
        this.complaintForm.patchValue(complaint);
      }
    });

    this.sub$.sink = this.route.queryParams.subscribe((params) => {
      if (params['isDetailsMode'] === 'true') {
        this.isDetailsMode = false;
      }
    })
  }

  createComplaintForm() {
    this.complaintForm = this.fb.group({
      id: [''],
      title: ['', [Validators.required]],
      complaintNo: [''],
      complaintDate: [new Date()],
      description: ['', [RichTextRequired]],
      reportedBy: ['', [Validators.required]],
      assignedToId: ['', [Validators.required]],
      complaintTypeId: ['', [Validators.required]],
      priority: ['', [Validators.required]],
      status: ['', [Validators.required]],
    })
  }

  getAllCompamintTypes() {
    this.sub$.sink = this.complaintTypeService.getComplaintTypes().subscribe({
      next: (complaintTypes) => {
        const types = complaintTypes as ComplaintType[];
        if (types) {
          this.complaintTypeList = [...types];
        }
      },
    })
  }

  onSubmit() {
    if (!this.complaintForm.valid) {
      this.complaintForm.markAllAsTouched();
      return;
    }

    const complaintData = this.complaintForm.getRawValue();

    if (this.isEdit) {
      this.complaintStore.updateComplaint(complaintData);
    } else {
      this.complaintStore.addComplaint(complaintData);
    }
  }

  subscribeIsAddorUpdate() {
    toObservable(this.complaintStore.isAddUpdate).subscribe((flag) => {
      if (flag) {
        const id = this.complaintForm.get('id')?.value;
        const route = this.isDetailsMode ? ['/complaint/list'] : [`complaint/details/${id}/action`];
        this.router.navigate(route);
        this.complaintStore.resetflag();
      }
    });
  }

  addComplaintType(): void {
    const dialogRef = this.dialog.open(ManageComplaintTypeComponent, {
      width: '500px',
    });
    dialogRef.afterClosed().subscribe((result: ComplaintType) => {
      if (result) {
        this.complaintTypeList = [...this.complaintTypeList, result];
        this.complaintForm.patchValue({ complaintTypeId: result.id });
      }
    });
  }
}
