import {
  Component,
  ElementRef,
  OnInit,
  Renderer2,
  HostListener,
  ChangeDetectorRef,
  Inject,
  DOCUMENT
} from '@angular/core';
import { MatButtonModule } from '@angular/material/button';
import { MatIconModule } from '@angular/material/icon';
import { MatMenuModule } from '@angular/material/menu';
import { Router, RouterModule } from '@angular/router';
import { LanguageFlag } from '@core/domain-classes/language-flag';
import { UserNotification } from '@core/domain-classes/notification';
import { UserAuth } from '@core/domain-classes/user-auth';
import { SecurityService } from '@core/security/security.service';
import { CategoryService } from '@core/services/category.service';
import { ClonerService } from '@core/services/clone.service';
import { CommonService } from '@core/services/common.service';
import { SignalrService } from '@core/services/signalr.service';
import { WINDOW, WINDOW_PROVIDERS } from '@core/services/window.service';
import { TranslateModule } from '@ngx-translate/core';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { NgScrollbarModule } from 'ngx-scrollbar';
import { BaseComponent } from '../../base.component';
import { NotificationService } from '../../notification/notification.service';
import { NgClass } from '@angular/common';
import { MatBadgeModule } from '@angular/material/badge';

@Component({
  selector: 'app-header',
  templateUrl: './header.component.html',
  styleUrls: ['./header.component.scss'],
  standalone: true,
  imports: [
    RouterModule,
    TranslateModule,
    MatMenuModule,
    NgScrollbarModule,
    MatIconModule,
    MatButtonModule,
    UTCToLocalTime,
    NgClass,
    MatBadgeModule
  ],
  providers: [
    WINDOW_PROVIDERS
  ]
})
export class HeaderComponent extends BaseComponent implements OnInit {
  logoUrl?: string;
  logoIconUrl?: string;
  isNavbarCollapsed = true;
  isNavbarShow = true;
  isOpenSidebar?: boolean;
  isFullScreen = false;
  docElement?: HTMLElement;
  appUserAuth!: UserAuth;
  newNotificationCount = 0;
  notifications: UserNotification[] = [];
  language!: LanguageFlag | undefined;
  direction = 'ltr';
  hasClass = false;
  oldLang: string = '';
  languages: LanguageFlag[] = [
    {
      code: 'en',
      name: 'English',
      imageUrl: '../../../images/flags/united-states.svg',
      active: true,
      isrtl: false
    },
    {
      code: 'es',
      name: 'Spanish ',
      imageUrl: '../../../images/flags/brazil.svg',
      active: true,
      isrtl: false
    },
    {
      code: 'fr',
      name: 'French ',
      imageUrl: '../../../images/flags/france.svg',
      active: true,
      isrtl: false
    },
    {
      code: 'ar',
      name: 'Arabic ',
      imageUrl: '../../../images/flags/saudi-arabia.svg',
      active: true,
      isrtl: true
    },
    {
      code: 'tr',
      name: 'Turkish',
      imageUrl: '../../../images/flags/turkish.png',
      active: true,
      isrtl: false
    },
    {
      code: 'pl',
      name: 'Polish',
      imageUrl: '../../../images/flags/polish.jpeg',
      active: true,
      isrtl: false
    },
    {
      code: 'ru',
      name: 'Russian',
      imageUrl: '../../../images/flags/russia.svg',
      active: true,
      isrtl: false
    },
    {
      code: 'ja',
      name: 'Japanese',
      imageUrl: '../../../images/flags/japan.svg',
      active: true,
      isrtl: false
    },
    {
      code: 'ko',
      name: 'Korean',
      imageUrl: '../../../images/flags/south-korea.svg',
      active: true,
      isrtl: false
    },
    {
      code: 'cn',
      name: 'Chinese',
      imageUrl: '../../../images/flags/china.svg',
      active: true,
      isrtl: false
    }
  ];
  constructor(
    @Inject(DOCUMENT) private document: Document,
    @Inject(WINDOW) private window: Window,
    private renderer: Renderer2,
    public elementRef: ElementRef,
    private router: Router,
    private securityService: SecurityService,
    private notificationService: NotificationService,
    private signalrService: SignalrService,
    private cd: ChangeDetectorRef,
    private commonService: CommonService,
    private clonerService: ClonerService,
    private categoryService: CategoryService,
  ) {
    super();
  }


  @HostListener('window:scroll', [])
  onWindowScroll() {
    this.window.pageYOffset ||
      this.document.documentElement.scrollTop ||
      this.document.body.scrollTop ||
      0;
  }

  ngOnInit() {
    this.docElement = document.documentElement;
    this.sidebarMenuStatus();
    this.setTopLogAndName();
    this.subscribeToNotification();
    this.setDefaultLanguage();
    this.getCompanyProfile();

  }


  sidebarMenuStatus() {
    this.sub$.sink = this.commonService.sideMenuStatus$.subscribe((status) => {
      this.isOpenSidebar = status;
    });
  }

  callFullscreen() {
    if (!this.isFullScreen) {
      if (this.docElement?.requestFullscreen != null) {
        this.docElement?.requestFullscreen();
      }
    } else {
      document.exitFullscreen();
    }
    this.isFullScreen = !this.isFullScreen;
  }

  mobileMenuSidebarOpen(event: Event, className: string) {
    const hasClass = (event.target as HTMLInputElement).classList.contains(
      className
    );
    if (hasClass) {
      this.renderer.removeClass(this.document.body, className);
    } else {
      this.renderer.addClass(this.document.body, className);
    }
  }

  callSidemenuCollapse() {
    const hasClass = this.document.body.classList.contains('side-closed');
    if (hasClass) {
      this.commonService.setSideMenuStatus(false);
      this.renderer.removeClass(this.document.body, 'side-closed');
      this.renderer.removeClass(this.document.body, 'submenu-closed');
      localStorage.setItem('collapsed_menu', 'false');
    } else {
      this.renderer.addClass(this.document.body, 'side-closed');
      this.renderer.addClass(this.document.body, 'submenu-closed');
      localStorage.setItem('collapsed_menu', 'true');
      this.commonService.setSideMenuStatus(true);
    }
  }

  setDefaultLanguage() {
    const lang = this.translationService.getSelectedLanguage();
    if (lang) this.setLanguageWithRefresh(lang);
  }

  setLanguageWithRefresh(code: string) {
    this.language = this.languages.find((c) => c.code == code);
    this.languages.forEach((language: LanguageFlag) => {
      if (language.code === code) {
        language.active = true;
      } else {
        language.active = false;
      }
    });
    if (this.language)
      this.translationService.setLanguage(this.language);
  }
  setLanguageWithRefreshNew(languageflag: LanguageFlag) {
    this.languages.forEach((language: LanguageFlag) => {
      if (language.code === languageflag.code) {
        language.active = true;
        this.language = languageflag;
      } else {
        languageflag.active = false;
      }
    });
    this.translationService.setLanguage(languageflag);
  }

  setNewLanguageRefresh(language: LanguageFlag) {
    this.oldLang = this.translationService.getSelectedLanguage();
    this.sub$.sink = this.translationService
      .setLanguage(language)
      .subscribe((response: any) => {
        if (response)
          this.setLanguageWithRefreshNew(language);
      });
  }

  setTopLogAndName() {
    this.sub$.sink = this.securityService.SecurityObject.subscribe((c) => {
      if (c) {
        this.appUserAuth = c;
      }
    });
  }

  onLogout(): void {
    this.securityService.logout();
    sessionStorage.removeItem('workflows');
    sessionStorage.removeItem(this.categoryService.categoryKey);
    location.reload();
  }

  onMyProfile(): void {
    this.router.navigate(['/my-profile']);
  }

  subscribeToNotification() {
    this.sub$.sink = this.signalrService.userNotification$.subscribe((c) => {
      this.getNotification();
    });
  }

  getNotification() {
    this.sub$.sink = this.notificationService
      .getNotification()
      .subscribe((notifications: UserNotification[]) => {
        const unreadCount = notifications.filter((c) => !c.isRead).length;
        this.newNotificationCount = unreadCount;
        this.notifications =
          this.clonerService.deepClone<UserNotification[]>(notifications);
        this.cd.detectChanges();
      });
  }


  markAllAsReadNotification() {
    this.sub$.sink = this.notificationService.markAllAsRead().subscribe(() => {
      this.getNotification();
    });
  }

  viewAllNotification() {
    this.sub$.sink = this.notificationService.markAllAsRead().subscribe(() => {
      this.router.navigate(['/notifications']);
    });
  }

  viewNotification(notification: UserNotification) {
    if (!notification.isRead) {
      this.markAsReadNotification(notification.id ?? '');
    }

    switch (notification.notificationsType) {
      case 0: // REMINDER
        this.router.navigate(['/reminders']);
        break;

      case 1: // SHARE_USER
        if (notification.documentId) {
          this.router.navigate(['/assign/list-view']);
        } else {
          this.router.navigate(['/assign/folder-view']);
        }
        break;

      case 2: // WORKFLOW
        this.router.navigate(['/current-workflow']);
        break;

      case 3: // FILE_REQUEST
        this.router.navigate(['/file-request']);
        break;

      case 4: // SHARE_FOLDER
        this.router.navigate(['/assign/folder-view']);
        break;

      case 5: // COURSE
        this.router.navigate(['/course/my-courses']);
        break;

      case 6: // AUDIT
        this.router.navigate(['/audit/audit-list']);
        break;

      case 7: // CAPA_ACTION
        this.router.navigate(['/capa/actions']);
        break;

      case 8: // COMPLAINT_ACTION
        this.router.navigate(['/complaint/actions']);
        break;

      case 9: // COMPLAINT_INVESTIGATION
        this.router.navigate(['/complaint/actions']);
        break;

      case 10: // NC_RESPONSE
        this.router.navigate(['/nc/conformances']);
        break;

      case 11: // RISK
        this.router.navigate(['/risk/list']);
        break;

      case 12: // RISK_CONTROL
        this.router.navigate(['/risk/my-risk-list']);
        break;

      case 13: // RISK_REVIEW
        this.router.navigate(['/risk/my-risk-list']);
        break;

      case 14: // COMPLAINT
        this.router.navigate(['/complaint/list']);
        break;

      case 15: // AUDITREWORK
        this.router.navigate(['/audit/audit-list']);
        break;

      case 16: // AUDITREJECTED
        this.router.navigate(['/audit/audit-list']);
        break;

      case 17: // AUDITINREVIEW
        this.router.navigate(['/audit/audit-for-review']);
        break;

      default:
        if (notification.documentId) {
          this.router.navigate(['/assign/list-view']);
        } else {
          this.router.navigate(['/']);
        }
        break;
    }
  }

  markAsReadNotification(id: string) {
    this.sub$.sink = this.notificationService.markAsRead(id).subscribe(() => {
      this.getNotification();
    });
  }

  getCompanyProfile(): void {
    this.securityService.companyProfile.subscribe((c) => {
      if (c) {
        this.logoUrl = c.logoUrl;
        this.logoIconUrl = c.logoIconUrl;
      }
    });
  }

  getNotificationMessage(notification: UserNotification): string {
    if (notification.jsonPayload) {
      const payload = JSON.parse(notification.jsonPayload);

      switch (notification.notificationsType) {
        case 5: // COURSE
          return this.translationService.getValue('YOU_HAVE_BEEN_ASSIGNED_THE_COURSE') + `"${payload.Title}"` + this.translationService.getValue('PLEASE_COMPLETE_IT_AS_SCHEDULED');

        case 6: // AUDIT
          return this.translationService.getValue('YOU_HAVE_BEEN_ASSIGNED_THE_AUDIT') + `"${payload.Title}".`;

        case 7: // CAPA_ACTION
          return this.translationService.getValue('YOU_HAVE_BEEN_ASSIGNED_A_CAPA_ACTION_FOR') + `"${payload.Title}".`;

        case 8: // COMPLAINT_ACTION
          return this.translationService.getValue('YOU_HAVE_BEEN_DESIGNATED_AS_THE_RESPONSIBLE_PERSON_FOR_THE_COMPLAINT_ACTION') + `"${payload.Title}".`;

        case 9: // COMPLAINT_INVESTIGATION
          return this.translationService.getValue('YOU_HAVE_BEEN_ASSIGNED_TO_INVESTIGATE_THE_COMPLAINT') + `"${payload.Title}".`;

        case 10: // NC_RESPONSE
          return this.translationService.getValue('YOU_HAVE_BEEN_DESIGNATED_AS_THE_RESPONSIBLE_PERSON_FOR_THE_NON_CONFORMANCE') + `"${payload.Title}".`;

        case 11: // RISK
          return this.translationService.getValue('YOU_HAVE_BEEN_ASSIGNED_TO_REVIEW_HANDLE_THE_RISK') + `"${payload.Title}".`;

        case 12: // RISK_CONTROL
          return this.translationService.getValue('YOU_HAVE_BEEN_DESIGNATED_AS_THE_RESPONSIBLE_PERSON_FOR_THE_RISK_CONTROL_TASK') + `"${payload.Title}".`;

        case 13: // RISK_REVIEW
          return this.translationService.getValue('YOU_HAVE_BEEN_DESIGNATED_AS_THE_REVIEWER_FOR_A_RISK') + `"${payload.Title}".` || payload.Message;

        case 14: // COMPLAINT
          return this.translationService.getValue('A_NEW_COMPLAINT') + `"${payload.Title}"` + this.translationService.getValue('HAS_BEEN_ASSIGNED_TO_YOU');

        case 15: // AUDITREWORK
          return this.translationService.getValue('THE_AUDIT') + `"${payload.Title}"` + this.translationService.getValue('HAS_BEEN_SENT_BACK_FOR_REWORK');

        case 16: // AUDITREJECTED
          return this.translationService.getValue('THE_AUDIT') + `"${payload.Title}"` + this.translationService.getValue('HAS_BEEN_REJECTED');

        case 17: // AUDITINREVIEW
          return this.translationService.getValue('THE_AUDIT') + `"${payload.Title}"` + this.translationService.getValue('HAS_BEEN_SUBMITTED_FOR_YOUR_REVIEW');

        default:
          return notification.message || notification.documentName || this.translationService.getValue('YOU_HAVE_A_NEW_NOTIFICATION');
      }
    } else {
      switch (notification.notificationsType) {
        case 0: // REMINDER
          return notification.message ? this.translationService.getValue('REMINDER') + ` :: "${notification.message}"` : this.translationService.getValue('YOU_HAVE_A_REMINDER_NOTIFICATION');

        case 1: // SHARE_USER
          return notification.message || notification.documentName || this.translationService.getValue('YOU_HAVE_BEEN_GRANTED_ACCESS_TO_A_FILE_OR_FOLDER');

        case 2: // WORKFLOW
          return notification.message || notification.documentName || this.translationService.getValue('A_WORKFLOW_TASK_HAS_BEEN_ASSIGNED_TO_YOU');

        case 3: // FILE_REQUEST
          return notification.message || notification.documentName || this.translationService.getValue('SOMEONE_HAS_REQUESTED_YOU_TO_UPLOAD_REQUIRED_FILES');

        case 4: // SHARE_FOLDER
          return notification.message || notification.documentName || this.translationService.getValue('A_FOLDER_HAS_BEEN_SHARED_WITH_YOU');
        default:
          return notification.message || notification.documentName || this.translationService.getValue('YOU_HAVE_A_NEW_NOTIFICATION');
      }
    }
  }
}
