import { Injectable } from '@angular/core';
import { Observable, BehaviorSubject, of } from 'rxjs';
import { HttpClient } from '@angular/common/http';
import { tap, catchError } from 'rxjs/operators';
import { UserAuth } from '../domain-classes/user-auth';
import { CommonHttpErrorService } from '../error-handler/common-http-error.service';
import { ClonerService } from '../services/clone.service';
import { CommonError } from '../error-handler/common-error';
import { User } from '@core/domain-classes/user';
import { Router } from '@angular/router';
import { CompanyProfile } from '../../company-profile/company-profile';
import { jwtDecode } from 'jwt-decode';
import { LicenseValidatorService } from '@mlglobtech/license-validator-qms';

@Injectable({ providedIn: 'root' })
export class SecurityService {
  securityObject: UserAuth = new UserAuth();
  private securityObject$: BehaviorSubject<UserAuth> =
    new BehaviorSubject<UserAuth>({} as UserAuth);

  private companyProfile$: BehaviorSubject<CompanyProfile> = new BehaviorSubject<CompanyProfile>({} as CompanyProfile);

  private _claims: { [key: string]: string } | null = null;

  public get Claims(): { [key: string]: string } | null {
    if (this._claims) {
      return this._claims;
    }

    const token = localStorage.getItem('bearerToken');
    if (token) {
      this._claims = jwtDecode(token);
    }
    return this._claims;
  }

  public get SecurityObject(): Observable<UserAuth> {
    return this.securityObject$.asObservable();
  }
  constructor(
    private http: HttpClient,
    private clonerService: ClonerService,
    private commonHttpErrorService: CommonHttpErrorService,
    private licenseValidatorService: LicenseValidatorService,
    private router: Router
  ) { }
  isUserAuthenticate(): boolean {
    if (this.securityObject.userName && this.securityObject.bearerToken) {
      return true;
    } else {
      return this.parseSecurityObj();
    }
  }
setCompany(companyProfile?: CompanyProfile) {
    if (companyProfile) {
      sessionStorage.setItem(
        this.licenseValidatorService.keyValues.COMPANY_PROFILE,
        JSON.stringify(companyProfile)
      );
      this.companyProfile$.next(JSON.parse(JSON.stringify(companyProfile)));
    } else {
      const companyProfileJson = sessionStorage.getItem(
        this.licenseValidatorService.keyValues.COMPANY_PROFILE
      );
      if (
        companyProfileJson &&
        companyProfileJson !== 'null' &&
        companyProfileJson !== 'undefined'
      ) {
        this.companyProfile$.next(JSON.parse(companyProfileJson));
      }
    }
  }

  public get companyProfile(): Observable<CompanyProfile> {
    return this.companyProfile$.asObservable();
  }

  login(entity: User): Observable<UserAuth> {
    // Initialize security object
    this.resetSecurityObject();
    return this.http
      .post<UserAuth>('user/login', entity)
      .pipe(
        tap((resp) => {
          this.securityObject = this.clonerService.deepClone<UserAuth>(resp);
          this.licenseValidatorService.setTokenValue(this.securityObject);
          this.securityObject$.next(resp);
        })
      );

  }
  refreshToken(): Observable<UserAuth | CommonError> {
    return this.http
      .get<UserAuth>('user/refresh_token')
      .pipe(
        tap((resp) => {
          this.securityObject = this.clonerService.deepClone<UserAuth>(resp);
          this.licenseValidatorService.setTokenValue(this.securityObject);
          this.securityObject$.next(resp);
        })
      )
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  // public parseSecurityObj(): boolean {
  //   const securityObjectString = this.licenseValidatorService.getAuthObject();
  //   if (!securityObjectString) {
  //     return false;
  //   }
  //   const secuObj = JSON.parse(securityObjectString);
  //   this.securityObject = this.clonerService.deepClone<UserAuth>(secuObj);
  //   if (this.securityObject.userName && this.securityObject.bearerToken) {
  //     this.securityObject$.next(this.securityObject);
  //     return true;
  //   }
  //   return false;
  // }
  public parseSecurityObj(): boolean {
  const securityObjectString = this.licenseValidatorService.getAuthObject();
  if (!securityObjectString) {
    return false;
  }

  try {
    // Handle both JSON string and already-parsed object
    const secuObj =
      typeof securityObjectString === 'string'
        ? JSON.parse(securityObjectString)
        : securityObjectString;

    this.securityObject = this.clonerService.deepClone<UserAuth>(secuObj);

    if (this.securityObject.userName && this.securityObject.bearerToken) {
      this.securityObject$.next(this.securityObject);
      return true;
    }
  } catch (err) {
    console.error('Invalid security object in parseSecurityObj():', err, securityObjectString);
    // remove the corrupted data to prevent infinite errors
    this.licenseValidatorService.removeToken();
  }

  return false;
}


  logout(): void {
    this.resetSecurityObject();
  }

  resetSecurityObject(): void {
    this.securityObject.userName = '';
    this.securityObject.bearerToken = '';
    this.securityObject.isAuthenticated = false;
    this.securityObject.firstName = '';
    this.securityObject.lastName = '';
    this.securityObject.claims = []; 
    this.licenseValidatorService.removeToken();
    sessionStorage.clear();
    this.securityObject$.next({} as UserAuth);
    this.router.navigate(['/login']);
  }

  // This method can be called a couple of different ways
  // *hasClaim="'claimType'"  // Assumes claimValue is true
  // *hasClaim="'claimType:value'"  // Compares claimValue to value
  // *hasClaim="['claimType1','claimType2:value','claimType3']"
  // tslint:disable-next-line: typedef
  hasClaim(claimType: any): boolean {
    let ret = false;
    // See if an array of values was passed in.
    if (typeof claimType === 'string') {
      ret = this.isClaimValid(claimType);
    } else {
      const claims: string[] = claimType;
      if (claims.length === 0) {
        return true;
      }
      if (claims) {
        // tslint:disable-next-line: prefer-for-of
        for (let index = 0; index < claims.length; index++) {
          ret = this.isClaimValid(claims[index]);
          // If one is successful, then let them in
          if (ret) {
            break;
          }
        }
      }
    }
    return ret;
  }

  // private isClaimValid(claimType: string, claimValue?: string): boolean {
  //   let ret = false;
  //   let auth: UserAuth = null;
  //   // Retrieve security object
  //   auth = this.securityObject;
  //   if (auth) {
  //     // See if the claim type has a value
  //     // *hasClaim="'claimType:value'"
  //     if (claimType.indexOf(':') >= 0) {
  //       const words: string[] = claimType.split(':');
  //       claimType = words[0].toLowerCase();
  //       claimValue = words[1];
  //     } else {
  //       claimType = claimType.toLowerCase();
  //       // Either get the claim value, or assume 'true'
  //       claimValue = claimValue ? claimValue : 'true';
  //     }
  //     // Attempt to find the claim
  //     ret =
  //       auth.claims.find(
  //         (c) =>
  //           c.claimType.toLowerCase() == claimType.toLowerCase() && c.claimValue == claimValue
  //       ) != null;
  //   }

  //   return ret;
  // }

  private isClaimValid(claimType: string, claimValue?: string): boolean {
    let ret = false;
    // See if the claim type has a value
    // *hasClaim="'claimType:value'"
    if (claimType.indexOf(':') >= 0) {
      const words: string[] = claimType.split(':');
      claimType = words[0].toLowerCase();
      claimValue = words[1];
    } else {
      claimType = claimType.toLowerCase();
      // Either get the claim value, or assume 'true'
      claimValue = claimValue ? claimValue : 'true';
    }
    const token = this.licenseValidatorService.getJWtToken();
    if (token) {
      const claims = Object.keys(token).filter((key) => token[key]);
      ret = claims?.find((c: any) => c.toLowerCase() == claimType) != null;
    }

    return ret;
  }

  getUserDetail(): UserAuth {
    var userJson = this.licenseValidatorService.getAuthObject();
    if (!userJson) {
      return new UserAuth();
    }
    return userJson as UserAuth;
  }

  setUserDetail(user: UserAuth) {
    this.securityObject = this.clonerService.deepClone<UserAuth>(user);
    this.licenseValidatorService.setTokenValue(this.securityObject);
  }

  getCompanyProfile(): Observable<CompanyProfile> {
    const url = `companyprofile`;
    return this.http
      .get<CompanyProfile>(url)
      ;
  }
  updateCompanyProfile(
    licenseKey: string,
    purchaseCode: string
  ): Observable<boolean | CommonError> {
    const url = `companyprofile/activate_license`;
    return this.http
      .post<boolean>(url, {
        purchaseCode: purchaseCode,
        licenseKey: licenseKey,
      })
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }
}

