import { Component, inject, OnInit } from '@angular/core';
import { debounceTime, distinctUntilChanged, Subject } from 'rxjs';
import { DashboradService } from '../dashboard.service';
import { BaseComponent } from '../../base.component';
import { HttpResponse } from '@angular/common/http';
import { CommonOverViewChart } from '../model/commonOverViewChart';
import { FormsModule } from '@angular/forms';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { NGX_ECHARTS_CONFIG, NgxEchartsModule } from 'ngx-echarts';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { TranslateModule } from '@ngx-translate/core';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';

@Component({
  selector: 'app-audit-count-by-department-chart',
  imports: [
    NgxEchartsModule,
    MatDatepickerModule,
    FormsModule,
    PageHelpTextComponent,
    TranslateModule,
    MatIconModule,
    MatCardModule
  ],
  providers: [
    {
      provide: NGX_ECHARTS_CONFIG,
      useValue: {
        echarts: () => import('echarts'),
      },
    }
  ],
  templateUrl: './audit-count-by-department-chart.component.html',
  styleUrl: './audit-count-by-department-chart.component.scss'
})
export class AuditCountByDepartmentChartComponent extends BaseComponent implements OnInit {
  dashboardService = inject(DashboradService);

  filterParameter$: Subject<string> = new Subject<string>();
  _FromDateFilter: Date | null = null;
  _ToDateFilter: Date | null = null;

  public get fromDateFilter(): Date | null {
    return this._FromDateFilter;
  }

  public set fromDateFilter(v: Date | null) {
    if (this._FromDateFilter !== v) {
      this._FromDateFilter = v;
      this.filterParameter$.next(`createdFromDate#${v ? v : ''}`);
    }
  }

  public get toDateFilter(): Date | null {
    return this._ToDateFilter;
  }

  public set toDateFilter(v: Date | null) {
    if (this._ToDateFilter !== v) {
      this._ToDateFilter = v;
      this.filterParameter$.next(`createdToDate#${v ? v : ''}`);
    }
  }

  ngOnInit(): void {
    this.geAuditCountByDepartMentChart();

    this.filterParameter$.pipe(
      distinctUntilChanged(),
      debounceTime(1000),
    ).subscribe((filter) => {
      const [key, value] = filter.split('#');
      if (key === 'createdFromDate') {
        this._FromDateFilter = value ? new Date(value) : null;
      }
      if (key === 'createdToDate') {
        this._ToDateFilter = value ? new Date(value) : null;
      }
      this.geAuditCountByDepartMentChart(
        this.fromDateFilter ?? undefined,
        this.toDateFilter ?? undefined
      );
    });
  }

  clearOrderDates() {
    this.fromDateFilter = null;
    this.toDateFilter = null;
  }

  geAuditCountByDepartMentChart(fromDate?: Date, toDate?: Date) {
    this.sub$.sink = this.dashboardService.getAuditByDepartment(fromDate, toDate).subscribe({
      next: (data) => {
        const response = data as HttpResponse<CommonOverViewChart[]>;
        if (response && response.body) {
          this.auditByDepartmentOptions = {
            ...this.auditByDepartmentOptions,
            series: [
              {
                ...this.auditByDepartmentOptions.series[0],
                data: response.body.map(item => ({
                  value: item.count,
                  name: item.name,
                })) ?? [],
              }
            ]
          };
        }
      },
    });
  }

  auditByDepartmentOptions: {
    tooltip: {
      trigger: string;
    },
    legend: {
      orient: string;
      left: string;
    },
    series: Array<{
      name: string;
      type: string;
      radius: string;
      data: Array<{ value: number; name: string }>;
      emphasis: {
        itemStyle: {
          shadowBlur: number;
          shadowOffsetX: number;
          shadowColor: string;
        }
      }
    }>
  } = {
      tooltip: {
        trigger: 'item'
      },
      legend: {
        orient: 'vertical',
        left: 'left'
      },
      series: [
        {
          name: 'Audit Status',
          type: 'pie',
          radius: '50%',
          data: [],
          emphasis: {
            itemStyle: {
              shadowBlur: 10,
              shadowOffsetX: 0,
              shadowColor: 'rgba(0, 0, 0, 0.5)'
            }
          }
        }
      ]
    };
}

