import { Component, inject, OnInit } from '@angular/core';
import { NGX_ECHARTS_CONFIG, NgxEchartsModule } from 'ngx-echarts';
import { DashboradService } from '../dashboard.service';
import { BaseComponent } from '../../base.component';
import { CommonOverViewChart } from '../model/commonOverViewChart';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { FormsModule } from '@angular/forms';
import { debounceTime, distinctUntilChanged, Subject } from 'rxjs';
import { HttpResponse } from '@angular/common/http';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { TranslateModule } from '@ngx-translate/core';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';

@Component({
  selector: 'app-audit-status-overview-chart',
  imports: [
    NgxEchartsModule,
    MatDatepickerModule,
    FormsModule,
    PageHelpTextComponent,
    TranslateModule,
    MatIconModule,
    MatCardModule
  ],
  providers: [
    {
      provide: NGX_ECHARTS_CONFIG,
      useValue: {
        echarts: () => import('echarts'),
      },
    }
  ],
  standalone: true,
  templateUrl: './audit-status-overview-chart.component.html',
  styleUrl: './audit-status-overview-chart.component.scss'
})
export class AuditStatusOverviewChartComponent extends BaseComponent implements OnInit {
  dashboardService = inject(DashboradService);

  filterParameter$: Subject<string> = new Subject<string>();
  _FromDateFilter: Date | null = null;
  _ToDateFilter: Date | null = null;

  public get fromDateFilter(): Date | null {
    return this._FromDateFilter;
  }

  public set fromDateFilter(v: Date | null) {
    if (this._FromDateFilter !== v) {
      this._FromDateFilter = v;
      this.filterParameter$.next(`createdFromDate#${v ? v : ''}`);
    }
  }

  public get toDateFilter(): Date | null {
    return this._ToDateFilter;
  }

  public set toDateFilter(v: Date | null) {
    if (this._ToDateFilter !== v) {
      this._ToDateFilter = v;
      this.filterParameter$.next(`createdToDate#${v ? v : ''}`);
    }
  }

  ngOnInit(): void {
    this.getAuditStatusOverviewChart();

    this.filterParameter$.pipe(
      distinctUntilChanged(),
      debounceTime(1000),
    ).subscribe((filter) => {
      const [key, value] = filter.split('#');
      if (key === 'createdFromDate') {
        this._FromDateFilter = value ? new Date(value) : null;
      }
      if (key === 'createdToDate') {
        this._ToDateFilter = value ? new Date(value) : null;
      }
      this.getAuditStatusOverviewChart(this.fromDateFilter ?? undefined, this.toDateFilter ?? undefined);
    });
  }

  clearOrderDates() {
    this.fromDateFilter = null;
    this.toDateFilter = null;
  }

  getAuditStatusOverviewChart(fromDate?: Date, toDate?: Date) {
    this.sub$.sink = this.dashboardService.getAuditStatusOverview(fromDate, toDate).subscribe({
      next: (data) => {
        const response = data as HttpResponse<CommonOverViewChart[]>;
        if (response && response.body) {
          this.auditStatusChartOptions = {
            ...this.auditStatusChartOptions,
            series: [
              {
                ...this.auditStatusChartOptions.series[0],
                data: response.body.map(item => ({
                  value: item.count,
                  name: this.translationService.getValue(item.name),
                })),
              }
            ]
          };
        }
      },
    });
  }

  auditStatusChartOptions: {
    tooltip: { trigger: string },
    legend: { orient: string, left: string },
    series: Array<{
      name: string,
      type: string,
      radius: string,
      data: Array<{ value: number, name: string }>,
      emphasis: {
        itemStyle: {
          shadowBlur: number,
          shadowOffsetX: number,
          shadowColor: string
        }
      }
    }>
  } = {
      tooltip: {
        trigger: 'item'
      },
      legend: {
        orient: 'vertical',
        left: 'left'
      },
      series: [
        {
          name: 'Audit Status',
          type: 'pie',
          radius: '50%',
          data: [],
          emphasis: {
            itemStyle: {
              shadowBlur: 10,
              shadowOffsetX: 0,
              shadowColor: 'rgba(0, 0, 0, 0.5)'
            }
          }
        }
      ]
    };
}
