import { Component, inject, OnInit } from '@angular/core';
import { FormsModule } from '@angular/forms';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { HttpResponse } from '@angular/common/http';
import { NGX_ECHARTS_CONFIG, NgxEchartsModule } from 'ngx-echarts';
import { Subject, distinctUntilChanged, debounceTime } from 'rxjs';
import { DashboradService } from '../dashboard.service';
import { CommonOverViewChart } from '../model/commonOverViewChart';
import { BaseComponent } from '../../base.component';
import { TranslateModule } from '@ngx-translate/core';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';

@Component({
  selector: 'app-capa-count-over-time-chart',
  imports: [
    MatDatepickerModule,
    FormsModule,
    NgxEchartsModule,
    TranslateModule,
    PageHelpTextComponent,
    MatIconModule,
    MatCardModule
  ],
  providers: [
    {
      provide: NGX_ECHARTS_CONFIG,
      useValue: {
        echarts: () => import('echarts'),
      },
    }
  ],
  templateUrl: './capa-count-over-time-chart.component.html',
  styleUrl: './capa-count-over-time-chart.component.scss'
})
export class CapaCountOverTimeChartComponent extends BaseComponent implements OnInit {
  dashboardService = inject(DashboradService);

  filterParameter$: Subject<string> = new Subject<string>();
  _FromDateFilter: Date | null = null;
  _ToDateFilter: Date | null = null;

  public get fromDateFilter(): Date | null {
    return this._FromDateFilter;
  }

  public set fromDateFilter(v: Date | null) {
    if (this._FromDateFilter !== v) {
      this._FromDateFilter = v;
      this.filterParameter$.next(`createdFromDate#${v ? v : ''}`);
    }
  }

  public get toDateFilter(): Date | null {
    return this._ToDateFilter;
  }

  public set toDateFilter(v: Date | null) {
    if (this._ToDateFilter !== v) {
      this._ToDateFilter = v;
      this.filterParameter$.next(`createdToDate#${v ? v : ''}`);
    }
  }

  ngOnInit(): void {
    this.getCapaChart();

    this.filterParameter$.pipe(
      distinctUntilChanged(),
      debounceTime(1000),
    ).subscribe((filter) => {
      const [key, value] = filter.split('#');
      if (key === 'createdFromDate') {
        this._FromDateFilter = value ? new Date(value) : null;
      }
      if (key === 'createdToDate') {
        this._ToDateFilter = value ? new Date(value) : null;
      }
      this.getCapaChart(this.fromDateFilter ?? undefined, this.toDateFilter ?? undefined);
    });
  }

  clearOrderDates() {
    this.fromDateFilter = null;
    this.toDateFilter = null;
  }

  getCapaChart(fromDate?: Date, toDate?: Date) {
    this.sub$.sink = this.dashboardService.getCapaChart(fromDate, toDate).subscribe({
      next: (data) => {
        const response = data as HttpResponse<CommonOverViewChart[]>;
        if (response && response.body) {
          const statusOrder = ["OPEN", "INPROGRESS", "CLOSED", "REJECTED", "REVIEWED"];
          const chartData = statusOrder.map(status => {
            const found = response.body?.find((item: CommonOverViewChart) => item.name === status);
            return found ? found.count : 0;
          });
          this.capaStatusChartOptions = {
            ...this.capaStatusChartOptions,
            series: [
              {
                ...this.capaStatusChartOptions.series[0],
                data: chartData,
              }
            ]
          };
        }
      },
    });
  }

  capaStatusChartOptions: any = {
    title: {
      text: '',
      left: 'center'
    },
    tooltip: {
      trigger: 'axis',
      axisPointer: {
        type: 'shadow'
      }
    },
    xAxis: {
      type: 'category',
      data: ['OPEN', 'INPROGRESS', 'CLOSED', 'REVIEWED', 'REJECTED'].map(item => this.translationService.getValue(item)),
      axisLabel: {
        rotate: 15
      }
    },
    yAxis: {
      type: 'value',
      min: 0,
      max: 30,
      interval: 5,
      axisLabel: {
        formatter: (value: number) => {
          return [5, 10, 15, 20, 25, 30].includes(value)
            ? value.toString()
            : '';
        },
      },
    },
    series: [
      {
        data: [] as number[],
        type: 'bar',
        itemStyle: {
          color: function (params: any) {
            const statusColorMap: Record<string, string> = {
              'Open': '#007bff',
              'In Progress': '#ffc107',
              'Closed': '#28a745',
              'Rejected': '#dc3545',
              'Reviewed': '#17a2b8' 
            };
            return statusColorMap[params.name] || '#888';
          }
        },
        label: {
          show: true,
          position: 'top'
        }
      }
    ]
  };
}
