import { Component, inject, OnInit } from '@angular/core';
import { FormsModule } from '@angular/forms';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { HttpResponse } from '@angular/common/http';
import { Subject, distinctUntilChanged, debounceTime } from 'rxjs';
import { BaseComponent } from '../../base.component';
import { DashboradService } from '../dashboard.service';
import { CommonOverViewChart } from '../model/commonOverViewChart';
import { NGX_ECHARTS_CONFIG, NgxEchartsModule } from 'ngx-echarts';
import { TranslateModule } from '@ngx-translate/core';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';

@Component({
  selector: 'app-nc-by-severity-level-chart',
  imports: [
    MatDatepickerModule,
    FormsModule,
    NgxEchartsModule,
    TranslateModule,
    PageHelpTextComponent,
    MatIconModule,
    MatCardModule
  ],
  providers: [
    {
      provide: NGX_ECHARTS_CONFIG,
      useValue: {
        echarts: () => import('echarts'),
      },
    }
  ],
  templateUrl: './nc-by-severity-level-chart.component.html',
  styleUrl: './nc-by-severity-level-chart.component.scss'
})
export class NcBySeverityLevelChartComponent extends BaseComponent implements OnInit {
  dashboardService = inject(DashboradService);

  filterParameter$: Subject<string> = new Subject<string>();
  _FromDateFilter: Date | null = null;
  _ToDateFilter: Date | null = null;

  public get fromDateFilter(): Date | null {
    return this._FromDateFilter;
  }

  public set fromDateFilter(v: Date | null) {
    if (this._FromDateFilter !== v) {
      this._FromDateFilter = v;
      this.filterParameter$.next(`createdFromDate#${v ? v : ''}`);
    }
  }

  public get toDateFilter(): Date | null {
    return this._ToDateFilter;
  }

  public set toDateFilter(v: Date | null) {
    if (this._ToDateFilter !== v) {
      this._ToDateFilter = v;
      this.filterParameter$.next(`createdToDate#${v ? v : ''}`);
    }
  }

  ngOnInit(): void {
    this.getNcSeverityChart();

    this.filterParameter$.pipe(
      distinctUntilChanged(),
      debounceTime(1000),
    ).subscribe((filter) => {
      const [key, value] = filter.split('#');
      if (key === 'createdFromDate') {
        this._FromDateFilter = value ? new Date(value) : null;
      }
      if (key === 'createdToDate') {
        this._ToDateFilter = value ? new Date(value) : null;
      }
      this.getNcSeverityChart(this.fromDateFilter, this.toDateFilter);
    });
  }

  clearOrderDates() {
    this._FromDateFilter = null;
    this._ToDateFilter = null;
  }

  getNcSeverityChart(fromDate?: Date | null, toDate?: Date | null): void {
    this.sub$.sink = this.dashboardService.getNcSeverityChart(fromDate, toDate).subscribe({
      next: (data) => {
        const response = data as HttpResponse<CommonOverViewChart[]>;
        if (response && response.body) {
          const statusOrder = ["LOW", "MEDIUM", "HIGH", "CRITICAL"];
          const chartData = statusOrder.map(status => {
            const found = response.body?.find((item: CommonOverViewChart) => item.name === status);
            return found ? found.count : 0;
          });
          this.ncSevertyLevelChartOptions = {
            ...this.ncSevertyLevelChartOptions,
            series: [
              {
                ...this.ncSevertyLevelChartOptions.series[0],
                data: chartData,
              }
            ]
          };
        }
      },
    });
  }

  ncSevertyLevelChartOptions: any = {
    title: {
      text: '',
      left: 'center'
    },
    tooltip: {
      trigger: 'axis',
      axisPointer: {
        type: 'shadow'
      }
    },
    xAxis: {
      type: 'category',
      data: ["LOW", "MEDIUM", "HIGH", "CRITICAL"].map(item => this.translationService.getValue(item)),
      axisLabel: {
        rotate: 15
      }
    },
    yAxis: {
      type: 'value',
      min: 0,
      max: 30,
      interval: 5,
      axisLabel: {
        formatter: (value: number) => {
          return [5, 10, 15, 20, 25, 30].includes(value)
            ? value.toString()
            : '';
        },
      },
    },
    series: [
      {
        data: [] as number[],
        type: 'bar',
        itemStyle: {
          color: function (params: any) {
            const statusColorMap: Record<string, string> = {
              'Low': '#28a745',
              'Medium': '#ffc107',
              'High': '#fd7e14',
              'Critical': '#dc3545'
            };
            return statusColorMap[params.name] || '#888';
          }
        },
        label: {
          show: true,
          position: 'top'
        }
      }
    ]
  };
}
